/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.model;

import java.util.List;

import org.fuin.jmsmvc4swing.base.Controller;
import org.fuin.jmsmvc4swing.jms.Data;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgConstructor;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgUtils;

/**
 * Generates a <code>ResultGenerator</code> implementation.
 * 
 * @param <I>
 *            The controller interface the created class is for.
 */
public class ResultGenerator<I extends Controller> {

    private final SgClassPool pool;

    private final String basePackage;

    private final String subPackage;

    private final String simpleDestClass;

    private final SgMethod intfListenerMethod;

    /**
     * Constructor.
     * 
     * @param pool
     *            Class pool to use.
     * @param model
     *            Controller model.
     * @param basePackage
     *            Base package.
     * @param intfListenerMethod
     *            Listener method from the interface the class depends on.
     */
    public ResultGenerator(final SgClassPool pool, final ControllerModel<I> model,
            final String basePackage, final SgMethod intfListenerMethod) {

        super();

        if (pool == null) {
            throw new IllegalArgumentException("The argument 'pool' cannot be null!");
        }
        this.pool = pool;

        if (model == null) {
            throw new IllegalArgumentException("The argument 'model' cannot be null!");
        }

        if (basePackage == null) {
            throw new IllegalArgumentException("The argument 'basePackage' cannot be null!");
        }
        this.basePackage = basePackage;

        final String listener = intfListenerMethod.getOwner().getSimpleNameAsPackage();
        final int pos = listener.lastIndexOf('_');
        if (pos == -1) {
            throw new IllegalStateException("Cannot find last '_' in '" + listener + "'!");
        }
        final String methodPackage = listener.substring(0, pos);

        this.subPackage = SgUtils.concatPackages(model.getPackageName(), SgUtils
                .concatPackages(methodPackage, intfListenerMethod.getNameAsPackage()));

        this.simpleDestClass = "Result";

        if (intfListenerMethod == null) {
            throw new IllegalArgumentException(
                    "The argument 'intfListenerMethod' cannot be null!");
        }
        this.intfListenerMethod = intfListenerMethod;

    }

    private String getDestPackage() {
        return SgUtils.concatPackages(basePackage, subPackage);
    }

    private void addFields(final SgClass clasz) {

        // Adds the field implicitly to the class
        final SgField serialField = new SgField(clasz, "private static", SgClass.LONG,
                "serialVersionUID");
        serialField.setInitializer("1L");

        final List<SgArgument> args = intfListenerMethod.getArguments();
        for (int i = 0; i < args.size(); i++) {
            final SgArgument arg = args.get(i);
            // Adds the field implicitly to the class
            final SgField field = new SgField(clasz, "private", arg.getType(), arg.getName());
            field.setInitializer("");
        }

    }

    private void addConstructor(final SgClass clasz) {

        // Adds the constructor implicitly to the class
        final SgConstructor constructor = new SgConstructor(clasz);
        
        final List<SgArgument> args = intfListenerMethod.getArguments();
        for (int i = 0; i < args.size(); i++) {
            // Adds the argument implicitly to the constructor
            new SgArgument(constructor, args.get(i));
        }
        
        constructor.addBodyLine("super();\n");
        for (int i = 0; i < args.size(); i++) {
            constructor.addBodyLine("this." + args.get(i).getName() + " = "
                    + args.get(i).getName() + ";");
        }

    }

    private void addGetters(final SgClass clasz) {

        final List<SgArgument> args = intfListenerMethod.getArguments();
        for (int i = 0; i < args.size(); i++) {
            final SgArgument arg = args.get(i);
            final SgMethod m = new SgMethod(clasz, "public", arg.getType(), "get"
                    + SgUtils.firstCharUpper(arg.getName()));
            m.addBodyLine("return " + arg.getName() + ";");
            clasz.addMethod(m);
        }

    }

    /**
     * Create the model class based on the interface informations.
     * 
     * @return Model class.
     */
    public final SgClass createModelClass() {

        // Create class definition
        final SgClass clasz = new SgClass("public", getDestPackage(), simpleDestClass, false,
                null);
        final SgClass intf = SgClass.create(pool, Data.class);
        clasz.addInterface(intf);

        addFields(clasz);
        addConstructor(clasz);
        addGetters(clasz);
        SgUtils.addToStringMethod(pool, clasz, clasz.getFields());

        pool.put(clasz);

        return clasz;
    }

}
