/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.model;

import java.util.List;

import org.fuin.jmsmvc4swing.base.Controller;
import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;
import org.fuin.jmsmvc4swing.jms.Data;
import org.fuin.jmsmvc4swing.jms.MethodHandler;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgConstructor;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgUtils;

/**
 * Generates a <code>MethodHandler</code> implementation.
 * 
 * @param <I>
 *            The controller interface the created class is for.
 */
public final class MethodHandlerGenerator<I extends Controller> extends
        AbstractMethodGenerator<I, MethodHandler> {

    /**
     * Constructor.
     * 
     * @param pool
     *            Class pool to use.
     * @param model
     *            Controller model.
     * @param basePackage
     *            Base package.
     * @param intfMethod
     *            Method from the interface the class depends on.
     */
    public MethodHandlerGenerator(final SgClassPool pool, final ControllerModel<I> model,
            final String basePackage, final SgMethod intfMethod) {

        super(pool, model, basePackage, intfMethod, "MethodHandler");

    }

    private void addFields(final SgClass clasz) {

        // Adds the field implicitly to the class
        final SgField field = new SgField(clasz, "private", SgClass.create(getPool(),
                JmsJndiEnvironment.class), "env");
        field.setInitializer("");

    }

    private void addConstructor(final SgClass clasz) {
        final SgConstructor constructor = new SgConstructor(clasz);
        // Adds the argument implicitly to the constructor
        new SgArgument(constructor, SgClass.create(getPool(), JmsJndiEnvironment.class), "env");
        constructor.addBodyLine("super();");
        constructor.addBodyLine("this.env = env;");
        clasz.addConstructor(constructor);
    }

    private void addMethods(final SgClass clasz) {

        final String pkg = clasz.getPackageName();

        // Add first method
        final SgMethod handleMethod = new SgMethod(clasz, "public", SgClass.VOID, "handle");

        // Adds the arguments implicitly to the method
        new SgArgument(handleMethod, SgClass.create(getPool(), Controller.class), "controller");
        new SgArgument(handleMethod, SgClass.create(getPool(), String.class), "topicName");
        new SgArgument(handleMethod, SgClass.create(getPool(), Data.class), "arguments");
        new SgArgument(handleMethod, SgClass.LONG, "id");

        handleMethod.addBodyLine(pkg + ".Arguments args = (" + pkg + ".Arguments) arguments;");
        handleMethod.addBodyLine(pkg + ".MethodResultSender resultPublisher = new " + pkg
                + ".MethodResultSender(env, topicName, " + pkg + ".Arguments.METHOD, id);");
        handleMethod.addBodyLine("resultPublisher.started();");
        handleMethod.addBodyLine("try {");
        handleMethod.addBodyLine("    ((" + getModel().getControllerInterface().getName()
                + ") controller)." + getIntfMethod().getName() + "("
                + createCommaSeparatedArgsGetter() + ", resultPublisher);");
        handleMethod.addBodyLine("    resultPublisher.succeded();");
        handleMethod.addBodyLine("} catch (RuntimeException ex) {");
        handleMethod.addBodyLine("    resultPublisher.failed(ex);");
        handleMethod.addBodyLine("}");
        clasz.addMethod(handleMethod);

        // Add second method
        final SgMethod getMethodNameMethod = new SgMethod(clasz, "public", SgClass.create(
                getPool(), String.class), "getMethodName");
        getMethodNameMethod.addBodyLine("return " + pkg + ".Arguments.METHOD;\n");
        clasz.addMethod(getMethodNameMethod);

    }

    private String createCommaSeparatedArgsGetter() {
        final StringBuffer sb = new StringBuffer();
        final List<SgArgument> args = getIntfMethod().getArguments();
        for (int i = 0; i < args.size() - 1; i++) {
            final SgArgument arg = args.get(i);
            if (i > 0) {
                sb.append(", ");
            }
            sb.append("args.get" + SgUtils.firstCharUpper(arg.getName()) + "()");
        }
        return sb.toString();
    }

    /**
     * Create the model class based on the interface informations.
     * 
     * @return Model class.
     */
    public final SgClass createModelClass() {

        // Create class definition
        final SgClass clasz = new SgClass("public", getDestPackage(), getSimpleDestClass(),
                false, null);
        final SgClass intf = SgClass.create(getPool(), MethodHandler.class);
        clasz.addInterface(intf);

        addFields(clasz);
        addConstructor(clasz);
        addMethods(clasz);

        getPool().put(clasz);

        return clasz;
    }

}
