/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.model;

import java.util.List;

import org.fuin.jmsmvc4swing.base.Controller;
import org.fuin.jmsmvc4swing.jms.Data;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgConstructor;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgUtils;

/**
 * Generates the "Argument" class.
 * 
 * @param <I>
 *            The controller interface the created class is for.
 */
public final class ArgumentsGenerator<I extends Controller> extends
        AbstractMethodGenerator<I, Data> {

    /**
     * Constructor.
     * 
     * @param pool
     *            Class pool to use.
     * @param model
     *            Controller model.
     * @param basePackage
     *            Base package.
     * @param intfMethod
     *            Method from the interface the class depends on.
     */
    public ArgumentsGenerator(final SgClassPool pool, final ControllerModel<I> model,
            final String basePackage, final SgMethod intfMethod) {

        super(pool, model, basePackage, intfMethod, "Arguments");

    }

    private void addFields(final SgClass clasz) {

        final SgField serialField = new SgField(clasz, "private static", SgClass.LONG,
                "serialVersionUID");
        serialField.setInitializer("1L");
        final SgField methodNameField = new SgField(clasz, "public static", SgClass.create(
                getPool(), String.class), "METHOD");
        methodNameField.setInitializer("\"" + getIntfMethod().getName() + "\"");
        final List<SgArgument> args = getIntfMethod().getArguments();
        // We don't want the last argument (always the listener) to be
        // serialized
        if (args.size() > 1) {
            for (int i = 0; i < args.size() - 1; i++) {
                final SgArgument arg = args.get(i);
                final SgField field = new SgField(clasz, "private", arg.getType(), arg
                        .getName());
                field.setInitializer("");
            }
        }
    }

    private void addConstructor(final SgClass clasz) {

        // Adds the constructor implicitly to the class
        final SgConstructor constructor = new SgConstructor(clasz);

        final List<SgArgument> arguments = getIntfMethod().getArguments(-1);
        for (int i = 0; i < arguments.size(); i++) {
            // Adds the argument implicitly to the constructor
            new SgArgument(constructor, arguments.get(i));
        }

        constructor.addBodyLine("super();\n");
        for (int i = 0; i < arguments.size(); i++) {
            constructor.addBodyLine("this." + arguments.get(i).getName() + " = "
                    + arguments.get(i).getName() + ";");
        }

    }

    private void addGetters(final SgClass clasz) {

        final List<SgArgument> args = getIntfMethod().getArguments();
        if (args.size() > 1) {
            for (int i = 0; i < args.size() - 1; i++) {
                final SgArgument arg = args.get(i);
                final SgMethod m = new SgMethod(clasz, "public", arg.getType(), "get"
                        + SgUtils.firstCharUpper(arg.getName()));
                m.addBodyLine("return " + arg.getName() + ";");
                clasz.addMethod(m);
            }
        }

    }

    /**
     * Create the model class based on the interface informations.
     * 
     * @return Model class.
     */
    public final SgClass createModelClass() {

        // Create class definition
        final SgClass clasz = new SgClass("public", getDestPackage(), getSimpleDestClass(),
                false, null);
        final SgClass intf = SgClass.create(getPool(), Data.class);
        clasz.addInterface(intf);

        addFields(clasz);
        addConstructor(clasz);
        addGetters(clasz);
        SgUtils.addToStringMethod(getPool(), clasz, clasz.getFields());

        getPool().put(clasz);

        return clasz;
    }

}
