/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.jms;

import javax.jms.JMSException;
import javax.jms.MessageListener;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;

import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;

/**
 * Handles a single JMS topic subscription.
 */
public class Subscriber {

    private final JmsJndiEnvironment env;

    private TopicSession session;

    private TopicSubscriber subscriber;

    /**
     * Constructor.
     * 
     * @param env
     *            Environment to use.
     */
    public Subscriber(final JmsJndiEnvironment env) {
        super();
        this.env = env;
    }

    /**
     * Subscribes to a topic with a selector based on the method arguments
     * (Opens a session to the JMS provider).
     * 
     * @param topicName
     *            Name of the topic to subscribe to.
     * @param methodName
     *            Name of the method for creating the selector.
     * @param id
     *            Id for creating the selector.
     * @param type
     *            Type for creating the selector.
     * @param listener
     *            Listener to handle the incoming messages.
     */
    public final void open(final String topicName, final String methodName, final Long id,
            final MessageType type, final MessageListener listener) {
        open(topicName, methodName, id, new MessageType[] { type }, listener);
    }

    /**
     * Subscribes to a topic with a selector based on the method arguments
     * (Opens a session to the JMS provider).
     * 
     * @param topicName
     *            Name of the topic to subscribe to.
     * @param methodName
     *            Name of the method for creating the selector.
     * @param id
     *            Id for creating the selector.
     * @param type
     *            Types for creating the selector.
     * @param listener
     *            Listener to handle the incoming messages.
     */
    public final void open(final String topicName, final String methodName, final Long id,
            final MessageType[] type, final MessageListener listener) {
        try {
            session = env.getConnection().createTopicSession(false, Session.AUTO_ACKNOWLEDGE);
            final Topic topic = env.getTopic(topicName);
            final String messageSelector = Utils.getMessageSelector(methodName, id, type);
            if (messageSelector == null) {
                subscriber = session.createSubscriber(topic);
            } else {
                subscriber = session.createSubscriber(topic, messageSelector, false);
            }
            subscriber.setMessageListener(listener);
        } catch (final JMSException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Returns if a JMS session is active.
     * 
     * @return If a subscription to a topic exists TRUE else FALSE.
     */
    public final boolean isOpen() {
        return (session != null);
    }

    /**
     * Returns if no JMS session is active.
     * 
     * @return If no subscription exists TRUE else FALSE.
     */
    public final boolean isClosed() {
        return (session == null);
    }

    /**
     * Unsubscribes to the topic (closes the session).
     */
    public final void close() {
        if (isOpen()) {
            try {
                session.close();
                session = null;
                subscriber = null;
            } catch (final JMSException ex) {
                throw new RuntimeException(ex);
            }
        }
    }

}
