/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.base;

import javax.swing.JComponent;

import org.divxdede.swing.busy.DefaultBusyModel;
import org.divxdede.swing.busy.JBusyComponent;

/**
 * Maps the received controller events to a <code>JBusyComponent</code>. The UI
 * will be locked when <code>started()</code> is called and unlocked when
 * <code>failed(Exception)</code> or <code>succeded()</code> is called. The
 * Adapter can be configured with the
 * <code>configure(Cancelable, boolean)</code> method to show a progress bar and
 * a "cancel" control.
 * 
 * @param <T>
 *            The type of the UI to be locked.
 */
public class ControllerResultBusyAdapter<T extends JComponent> implements
        ControllerResultListener, BusyConfigurable {

    private final JBusyComponent<T> busyComponent;

    /**
     * Constcructor with the UI to be locked.
     * 
     * @param busyComponent
     *            Component to be locked.
     */
    public ControllerResultBusyAdapter(final JBusyComponent<T> busyComponent) {
        super();
        this.busyComponent = busyComponent;
        busyComponent.setBusyModel(createDefaultBusyModel(false));
    }

    /**
     * Creates a busy model without a "cancel" control.
     * 
     * @param determinate
     *            If a progress bar should be visible TRUE else FALSE.
     * 
     * @return New model instance.
     */
    private DefaultBusyModel createDefaultBusyModel(final boolean determinate) {
        final DefaultBusyModel busyModel = new DefaultBusyModel();
        busyModel.setCancellable(false);
        busyModel.setAutoCompletionEnabled(false);
        busyModel.setDeterminate(false);
        return busyModel;
    }

    /**
     * Changes the underlying busy model.
     * 
     * @param cancelable
     *            The instance to be cancelled if the "cancel" UI control is
     *            pressed.
     * @param determinate
     *            If a progress bar should be visible TRUE else FALSE.
     */
    public final void configure(final Cancelable cancelable, final boolean determinate) {
        final DefaultBusyModel busyModel;
        if (cancelable == null) {
            busyModel = createDefaultBusyModel(determinate);
        } else {
            busyModel = new DefaultBusyModel() {
                private static final long serialVersionUID = 1L;

                @Override
                public void cancel() {
                    cancelable.cancel();
                    super.cancel();
                }
            };
            busyModel.setCancellable(true);
            busyModel.setDeterminate(determinate);
        }
        busyComponent.setBusyModel(busyModel);
    }

    /**
     * {@inheritDoc}
     */
    public final void started() {
        busyComponent.getBusyModel().setBusy(true);
    }

    /**
     * {@inheritDoc}
     */
    public final void progress(final ProgressInfo progressInfo) {
        busyComponent.getBusyModel().setRangeProperties(progressInfo.getValue(),
                progressInfo.getExtent(), progressInfo.getMinimum(),
                progressInfo.getMaximum(), false);
    }

    /**
     * {@inheritDoc}
     */
    public final void failed(final Exception ex) {
        busyComponent.getBusyModel().setBusy(false);
    }

    /**
     * {@inheritDoc}
     */
    public final void succeded() {
        busyComponent.getBusyModel().setBusy(false);
    }

}
