/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.model;

import java.util.List;

import org.fuin.jmsmvc4swing.base.Controller;
import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;
import org.fuin.jmsmvc4swing.jms.Data;
import org.fuin.jmsmvc4swing.jms.MethodResultReceiver;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgConstructor;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgUtils;

/**
 * Generates a <code>MethodResultReceiver</code> implementation.
 * 
 * @param <I>
 *            The controller interface the created class is for.
 */
public final class MethodResultReceiverGenerator<I extends Controller> extends
        AbstractMethodGenerator<I, MethodResultReceiver> {

    private final SgClass listenerClass;

    /**
     * Constructor.
     * 
     * @param pool
     *            Class pool to use.
     * @param model
     *            Controller model.
     * @param basePackage
     *            Base package.
     * @param intfMethod
     *            Method from the interface the class depends on.
     */
    public MethodResultReceiverGenerator(final SgClassPool pool,
            final ControllerModel<I> model, final String basePackage, final SgMethod intfMethod) {

        super(pool, model, basePackage, intfMethod, "MethodResultReceiver");

        final String intfName = model.getControllerInterface().getName();
        final String methodName = intfMethod.getName();
        final String listenerClassName = intfName + "$" + SgUtils.firstCharUpper(methodName)
                + "Listener";
        listenerClass = SgClass.create(getPool(), listenerClassName);

    }

    private void addFields(final SgClass clasz) {
        final SgField serialField = new SgField("private", listenerClass, "listener");
        serialField.setInitializer("");
        clasz.addField(serialField);
    }

    private void addConstructor(final SgClass clasz) {

        final SgConstructor constructor = new SgConstructor(clasz);
        constructor.addArgument(new SgArgument(SgClass.create(getPool(),
                JmsJndiEnvironment.class), "env"));
        constructor.addArgument(new SgArgument(listenerClass, "listener"));
        constructor.addArgument(new SgArgument(SgClass.create(getPool(), String.class),
                "topicName"));
        constructor.addArgument(new SgArgument(SgClass.create(getPool(), String.class),
                "methodName"));
        constructor.addArgument(new SgArgument(SgClass.LONG, "id"));
        constructor.addBodyLine("super(env, topicName, methodName, id, listener);");
        constructor.addBodyLine("this.listener = listener;");
        clasz.addConstructor(constructor);

    }

    private void addMethods(final SgClass clasz) {
        final SgMethod method = new SgMethod(clasz, "public", SgClass.VOID, "handleResult");
        method.addArgument(new SgArgument(SgClass.create(getPool(), Data.class), "result"));
        method.addBodyLine("try {");
        final List<SgMethod> listenerMethods = listenerClass.getMethods();
        for (int i = 0; i < listenerMethods.size(); i++) {
            final SgMethod listenerMethod = listenerMethods.get(i);
            final String pkg = getDestPackage() + "." + listenerMethod.getNameAsPackage();
            if (i == 0) {
                method.addBodyLine("    if (result instanceof " + pkg + ".Result) {");
            } else {
                method.addBodyLine("    } else if (result instanceof " + pkg + ".Result) {");
            }
            method.addBodyLine("        final " + pkg + ".Result resp = (" + pkg
                    + ".Result) result;");
            method.addBodyLine("        listener." + listenerMethod.getName() + "("
                    + getRespArgs(listenerMethod) + ");");
        }
        method.addBodyLine("    }");
        method.addBodyLine("} catch (RuntimeException ex) {");
        method.addBodyLine("    // TODO Handle exceptions!");
        method.addBodyLine("    ex.printStackTrace();");
        method.addBodyLine("}");
        clasz.addMethod(method);
    }

    private String getRespArgs(final SgMethod listenerMethod) {
        final StringBuffer sb = new StringBuffer();
        final List<SgArgument> args = listenerMethod.getArguments();
        for (int i = 0; i < args.size(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            final SgArgument arg = args.get(i);
            final String getter = "get" + SgUtils.firstCharUpper(arg.getName() + "()");
            sb.append("resp.");
            sb.append(getter);
        }
        return sb.toString();
    }

    /**
     * Create the model class based on the interface informations.
     * 
     * @return Model class.
     */
    public final SgClass createModelClass() {

        // Create class definition
        final SgClass clasz = new SgClass("public", getDestPackage(), getSimpleDestClass(),
                false, null);
        final SgClass superClass = SgClass.create(getPool(), MethodResultReceiver.class);
        clasz.setSuperClass(superClass);

        addFields(clasz);
        addConstructor(clasz);
        addMethods(clasz);

        getPool().put(clasz);

        return clasz;
    }

}
