/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.model;

import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

import org.fuin.jmsmvc4swing.base.Cancelable;
import org.fuin.jmsmvc4swing.base.Controller;
import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;
import org.fuin.jmsmvc4swing.base.MethodCancelable;
import org.fuin.jmsmvc4swing.base.ProgressDeterminate;
import org.fuin.jmsmvc4swing.jms.MethodCallSender;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgConstructor;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;

/**
 * Generates a <code>MethodCallSender</code> implementation.
 * 
 * @param <I>
 *            The controller interface the created class is for.
 */
public final class MethodCallSenderGenerator<I extends Controller> extends
        AbstractMethodGenerator<I, MethodCallSender> {

    /**
     * Constructor.
     * 
     * @param pool
     *            Class pool to use.
     * @param model
     *            Controller model.
     * @param basePackage
     *            Base package.
     * @param intfMethod
     *            Method from the interface the class depends on.
     */
    public MethodCallSenderGenerator(final SgClassPool pool, final ControllerModel<I> model,
            final String basePackage, final SgMethod intfMethod) {

        super(pool, model, basePackage, intfMethod, "MethodCallSender");

    }

    private void addFields(final SgClass clasz) {

        final SgField countField = new SgField("private static", SgClass.create(getPool(),
                AtomicLong.class), "COUNT");
        countField.setInitializer("new " + AtomicLong.class.getName() + "()");
        clasz.addField(countField);
        final SgField receiverField = new SgField("private", SgClass.create(getPool(),
                getDestPackage() + ".MethodResultReceiver"), "resultReceiver");
        receiverField.setInitializer("");
        clasz.addField(receiverField);

    }

    private void addConstructor(final SgClass clasz) {

        final SgConstructor constructor = new SgConstructor(clasz);
        constructor.addArgument(new SgArgument(SgClass.create(getPool(),
                JmsJndiEnvironment.class), "env"));
        constructor.addArgument(new SgArgument(SgClass.create(getPool(), String.class),
                "topicName"));
        constructor.addBodyLine("super(env, topicName, "
                + getIntfMethod().hasAnnotation(ProgressDeterminate.class.getName()) + ");");
        clasz.addConstructor(constructor);

    }

    private void addMethods(final SgClass clasz) {

        final List<SgArgument> intfArguments = getIntfMethod().getArguments();
        final SgArgument lastIntfArgument = intfArguments.get(intfArguments.size() - 1);
        final SgMethod method = new SgMethod(clasz, "public", SgClass.VOID, getIntfMethod()
                .getName());

        for (int i = 0; i < intfArguments.size(); i++) {
            method.addArgument(new SgArgument(intfArguments.get(i)));
        }

        method.addBodyLine("final long id = COUNT.incrementAndGet();\n");
        method.addBodyLine("resultReceiver = new " + getDestPackage()
                + ".MethodResultReceiver(getEnvironment(), " + lastIntfArgument.getName()
                + ", getTopicName(), " + getDestPackage() + ".Arguments.METHOD, id);\n");
        method.addBodyLine("publish(" + getDestPackage() + ".Arguments.METHOD, new "
                + getDestPackage() + ".Arguments("
                + getIntfMethod().getCommaSeparatedArgumentNames(-1)
                + "), id, resultReceiver);\n");
        clasz.addMethod(method);

        if (getIntfMethod().hasAnnotation(MethodCancelable.class.getName())) {
            final SgMethod cancelMethod = new SgMethod(clasz, "public", SgClass.VOID, "cancel");
            cancelMethod.addBodyLine("if (resultReceiver != null) {\n");
            cancelMethod.addBodyLine("    resultReceiver.cancel();\n");
            cancelMethod.addBodyLine("}\n");
            clasz.addMethod(cancelMethod);
        }

    }

    /**
     * Create the model class based on the interface informations.
     * 
     * @return Model class.
     */
    public final SgClass createModelClass() {

        // Create class definition
        final SgClass clasz = new SgClass("public", getDestPackage(), getSimpleDestClass(),
                false, null);
        final SgClass parent = SgClass.create(getPool(), MethodCallSender.class);
        clasz.setSuperClass(parent);
        if (getIntfMethod().hasAnnotation(MethodCancelable.class.getName())) {
            final SgClass cancelableIntf = SgClass.create(getPool(), Cancelable.class);
            clasz.addInterface(cancelableIntf);
        }

        addFields(clasz);
        addConstructor(clasz);
        addMethods(clasz);

        getPool().put(clasz);

        return clasz;
    }

}
