/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.model;

import java.util.List;

import org.fuin.jmsmvc4swing.base.Controller;
import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;
import org.fuin.jmsmvc4swing.jms.ControllerSender;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgConstructor;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgUtils;

/**
 * Generates a <code>ControllerSender</code> implementation.
 * 
 * @param <I>
 *            The controller interface the created class is for.
 */
public final class ControllerSenderGenerator<I extends Controller> extends
        AbstractGenerator<I, ControllerSender> {

    /**
     * Constructor.
     * 
     * @param pool
     *            Class pool to use.
     * @param model
     *            Controller model.
     * @param basePackage
     *            Base package.
     */
    public ControllerSenderGenerator(final SgClassPool pool, final ControllerModel<I> model,
            final String basePackage) {
        super(pool, model, basePackage, model.getPackageName(), model.getSimpleName()
                + "Sender");
    }

    private void setConstructorSrc(final SgConstructor constructor) {
        final List<SgMethod> methods = getModel().getControllerInterface().getMethods();
        for (int i = 0; i < methods.size(); i++) {
            final SgMethod method = methods.get(i);
            final String methodPackage = SgUtils.uppercaseToUnderscore(method.getName());
            constructor.addBodyLine(method.getName() + "CallSender = new " + getDestPackage()
                    + "." + methodPackage + ".MethodCallSender(env, topicName);");
        }
    }

    private void addMethod(final SgClass clasz, final SgMethod intfMethod) {

        final SgMethod method = new SgMethod(clasz, "public", SgClass.VOID, intfMethod
                .getName());
        final List<SgArgument> intfArguments = intfMethod.getArguments();
        for (int i = 0; i < intfArguments.size(); i++) {
            method.addArgument(new SgArgument(intfArguments.get(i)));
        }
        method.addBodyLine("configureBusyMode(" + method.getName() + "CallSender, "
                + intfArguments.get(intfArguments.size() - 1).getName() + ");");
        method.addBodyLine(method.getName() + "CallSender." + method.getName() + "("
                + intfMethod.getCommaSeparatedArgumentNames() + ");");
        clasz.addMethod(method);

    }

    /**
     * Create the model class based on the interface informations.
     * 
     * @return Model class.
     */
    public final SgClass createModelClass() {

        // Create class definition
        final SgClass clasz = new SgClass("public", getDestPackage(), getSimpleDestClass(),
                false, null);
        final SgClass parent = SgClass.create(getPool(), ControllerSender.class);
        clasz.setSuperClass(parent);
        clasz.addInterface(getModel().getControllerInterface());

        // Add field definitions
        final List<SgMethod> intfMethods = getModel().getControllerInterface().getMethods();
        for (int i = 0; i < intfMethods.size(); i++) {
            final SgMethod method = intfMethods.get(i);
            final String packageName = SgUtils.concatPackages(getDestPackage(), method
                    .getNameAsPackage());
            final SgClass type = SgClass.create(getPool(), packageName + ".MethodCallSender");
            final String name = method.getName() + "CallSender";
            final SgField field = new SgField("private", type, name);
            field.setInitializer("");
            clasz.addField(field);
        }

        // Add constructor definition
        final SgConstructor constructor = new SgConstructor(clasz);
        constructor.addArgument(new SgArgument(SgClass.create(getPool(),
                JmsJndiEnvironment.class), "env"));
        constructor.addArgument(new SgArgument(SgClass.create(getPool(), String.class),
                "topicName"));
        setConstructorSrc(constructor);
        clasz.addConstructor(constructor);

        // Add method definitions
        for (int i = 0; i < intfMethods.size(); i++) {
            addMethod(clasz, intfMethods.get(i));
        }

        getPool().put(clasz);

        return clasz;
    }

}
