/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.jmsmvc4swing.jms;

/**
 * Utilities for the package.
 */
public final class Utils {

    /**
     * Private constructor.
     */
    private Utils() {
        throw new UnsupportedOperationException("Cannot create an instance of this class!");
    }

    /**
     * Create a selector message for a given message type.
     * 
     * @param type
     *            Message type.
     * 
     * @return A string that can be used for a selector.
     */
    public static String getTypeExpr(final MessageType type) {
        return Constants.TYPE + "='" + type.getName() + "'";
    }

    /**
     * Create a selector message for given message types.
     * 
     * @param type
     *            Message types.
     * 
     * @return A string that can be used for a selector.
     */
    public static String getTypeExpr(final MessageType[] type) {
        if ((type == null) || (type.length == 0)) {
            return "";
        }
        if (type.length == 1) {
            return getTypeExpr(type[0]);
        }
        final StringBuffer sb = new StringBuffer();
        sb.append("(");
        for (int i = 0; i < type.length; i++) {
            if (i > 0) {
                sb.append(" OR ");
            }
            sb.append(getTypeExpr(type[i]));
        }
        sb.append(")");
        return sb.toString();
    }

    /**
     * Create a selector message for a given method name.
     * 
     * @param methodName
     *            Name of a method.
     * 
     * @return A string that can be used for a selector.
     */
    public static String getMethodExpr(final String methodName) {
        return Constants.METHOD + "='" + methodName + "'";
    }

    /**
     * Create a selector message for a given method call id.
     * 
     * @param id
     *            Unique method call id.
     * 
     * @return A string that can be used for a selector.
     */
    public static String getIdExpr(final long id) {
        return Constants.ID + "='" + id + "'";
    }

    /**
     * Create a selector message for a given method name, id and type.
     * 
     * @param methodName
     *            Name of a method - Can be null.
     * @param id
     *            Unique method call id - Can be null.
     * @param type
     *            Message type - Can be NULL.
     * 
     * @return A string that can be used for a selector.
     */
    public static String getMessageSelector(final String methodName, final Long id,
            final MessageType type) {
        return getMessageSelector(methodName, id, new MessageType[] { type });
    }

    /**
     * Create a selector message for a given method name, id and types.
     * 
     * @param methodName
     *            Name of a method - Can be null.
     * @param id
     *            Unique method call id - Can be null.
     * @param type
     *            Message types - Can be NULL.
     * 
     * @return A string that can be used for a selector.
     */
    public static String getMessageSelector(final String methodName, final Long id,
            final MessageType[] type) {

        final StringBuffer sb = new StringBuffer();
        if (methodName != null) {
            append(sb, getMethodExpr(methodName));
        }
        if (id != null) {
            append(sb, getIdExpr(id));
        }
        if (type != null) {
            append(sb, getTypeExpr(type));
        }
        return sb.toString();

    }

    /**
     * Append an expression to the selector string added with "AND".
     * 
     * @param sb
     *            String buffer to use.
     * @param expr
     *            Expression to append with "AND" expression
     */
    private static void append(final StringBuffer sb, final String expr) {
        if (sb.length() > 0) {
            sb.append(" AND ");
        }
        sb.append(expr);
    }

}
