/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.examples.utils4j;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.commons.io.IOUtils;
import org.fuin.utils4j.Utils4J;

/**
 * Example for zipping a directory.
 */
public final class ZipDirExample {

	/**
	 * Private constructor.
	 */
	private ZipDirExample() {
		throw new UnsupportedOperationException(
				"It's not allowed to create an instance of this class!");
	}

	private static void copyResourceToFile(final String resource,
			final File target) throws IOException {
		final InputStream in = ZipDirExample.class
				.getResourceAsStream(resource);
		try {
			final OutputStream out = new FileOutputStream(target);
			try {
				IOUtils.copy(in, out);
			} finally {
				out.close();
			}
		} finally {
			in.close();
		}
	}

	/**
	 * Executes the example.
	 * 
	 * @param args
	 *            Not used.
	 * 
	 * @throws IOException
	 *             Error creating the archive.
	 */
	public static void main(final String[] args) throws IOException {

		// Unpack example ZIP file
		final File zipFile = File.createTempFile("ZipDirExample", ".zip");
		try {
			copyResourceToFile("/myfile.zip", zipFile);
			final File zipDir = new File("c:/mydir");
			zipDir.mkdir();
			Utils4J.unzip(zipFile, zipDir);
			
		} finally {
			zipFile.delete();
		}

		// Given a local directory "mydir" on drive "c:" that contains
		// three files "x.txt", "y.txt" and "z.txt"
		final File dirToZip = new File("c:/mydir");
		
		// Run the ZIP command
		Utils4J.zipDir(dirToZip, "abc/def", new File("c:/myfile1.zip"));
		// The archive will contain the following structure:
		// abc/def/x.txt
		// abc/def/y.txt
		// abc/def/z.txt

		// Run the ZIP command again
		Utils4J.zipDir(dirToZip, "", new File("c:/myfile2.zip"));
		// The second archive will contain the following structure:
		// x.txt
		// y.txt
		// z.txt

	}
}
