/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.examples.jmsmvc4swing;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import java.util.Locale;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import org.divxdede.swing.busy.JBusyComponent;
import org.fuin.aspects4swing.InvokeAndWait;
import org.fuin.jmsmvc4swing.base.ControllerResultBusyAdapter;
import org.fuin.jmsmvc4swing.base.View;
import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.utils4j.Utils4J;
import org.fuin.utils4swing.glazedlists.TableFormatCreator;
import org.fuin.utils4swing.glazedlists.TableFormatExt;

import ca.odell.glazedlists.BasicEventList;
import ca.odell.glazedlists.SortedList;
import ca.odell.glazedlists.gui.AbstractTableComparatorChooser;
import ca.odell.glazedlists.swing.EventSelectionModel;
import ca.odell.glazedlists.swing.EventTableModel;
import ca.odell.glazedlists.swing.TableComparatorChooser;

import com.jeta.forms.components.panel.FormPanel;

/**
 * Form list panel (View).
 */
public final class AddressListPanel extends FormPanel implements View {

	private static final long serialVersionUID = 1L;

	private final SortedList<Address> addresses;

	private final EventSelectionModel<Address> addressSelection;

	private final JBusyComponent<AddressListPanel> busyComponent;

	private AddressController addressController;

	/**
	 * Constructor with class pool and generator.
	 * 
	 * @param pool
	 *            Class pool.
	 * @param generator
	 *            Generator.
	 */
	public AddressListPanel(final SgClassPool pool,
			final ByteCodeGenerator generator) {
		super(Utils4J.getPackagePath(AddressListPanel.class)
				+ "/AddressListPanel.jfrm");
		addresses = new SortedList<Address>(new BasicEventList<Address>());
		addressSelection = new EventSelectionModel<Address>(addresses);
		init(pool, generator);
		busyComponent = new JBusyComponent<AddressListPanel>();
		busyComponent.setView(this);
	}

	/**
	 * Initializes the UI.
	 * 
	 * @param classPool
	 *            Class pool.
	 * @param generator
	 *            Byte code generator for table format.
	 * 
	 */
	private void init(final SgClassPool classPool,
			final ByteCodeGenerator generator) {

		// Common
		setPreferredSize(new Dimension(600, 600));

		// Table
		final TableFormatCreator<Address> creator = new TableFormatCreator<Address>(
				classPool, generator);
		final TableFormatExt<Address> tableFormat = creator.create(
				Address.class, Locale.ENGLISH);
		final JTable table = getTableAddresses();
		table.setModel(new EventTableModel<Address>(addresses, tableFormat));
		table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		final TableColumnModel colModel = table.getColumnModel();
		for (int i = 0; i < tableFormat.getColumnCount(); i++) {
			final TableColumn col = colModel.getColumn(i);
			col.setPreferredWidth(tableFormat.getColumnWidth(i));
		}
		table.setSelectionModel(addressSelection);
		TableComparatorChooser.install(table, addresses,
				AbstractTableComparatorChooser.SINGLE_COLUMN);

		table.getSelectionModel().addListSelectionListener(
				new ListSelectionListener() {
					public void valueChanged(final ListSelectionEvent e) {
						final int selCount = table.getSelectedRowCount();
						getButtonEdit().setEnabled((selCount == 1));
						getButtonDelete().setEnabled((selCount > 0));
					}
				});

		getButtonAdd().addActionListener(new ActionListener() {
			public void actionPerformed(final ActionEvent e) {
				showNotImplementedDialog();
			}
		});

		getButtonEdit().addActionListener(new ActionListener() {
			public void actionPerformed(final ActionEvent e) {
				showNotImplementedDialog();
			}
		});

		getButtonDelete().addActionListener(new ActionListener() {
			public void actionPerformed(final ActionEvent e) {
				final Address address = getSelectedAddress();
				if (address != null) {

					class DeleteListener extends
							ControllerResultBusyAdapter<AddressListPanel>
							implements AddressController.DeleteAddressListener {

						public DeleteListener() {
							super(busyComponent);
						}

						@InvokeAndWait
						public void addressDeleted(final int id) {
							System.err.println("deleteAddress(int,long) ["
									+ Thread.currentThread() + "]");
							AddressListPanel.this.deleteAddress(id);
						}

					}
					addressController.deleteAddress(address,
							new DeleteListener());
				}
			}

		});

		getButtonRefresh().addActionListener(new ActionListener() {
			public void actionPerformed(final ActionEvent e) {

				class LoadListener extends
						ControllerResultBusyAdapter<AddressListPanel> implements
						AddressController.LoadAddressesListener {

					public LoadListener() {
						super(busyComponent);
					}

					@InvokeAndWait
					public void setAddresses(final List<Address> list) {
						System.err.println("setAdresses(List<Address>) ["
								+ Thread.currentThread() + "]");
						AddressListPanel.this.setAdresses(list);
					}

					@InvokeAndWait
					public void selectAddress(final int id) {
						System.err.println("selectAddress(int) ["
								+ Thread.currentThread() + "]");
						AddressListPanel.this.selectAdress(id);
					}

				}

				final Address address = getSelectedAddress();
				final int id;
				if (address == null) {
					id = -1;
				} else {
					id = address.getId();
				}

				addressController.loadAddresses(id, new LoadListener());

			}
		});

	}

	/**
	 * {@inheritDoc}
	 */
	public JComponent getView() {
		return busyComponent;
	}

	private JButton getButtonAdd() {
		return (JButton) getButton("ButtonAdd");
	}

	private JButton getButtonEdit() {
		return (JButton) getButton("ButtonEdit");
	}

	private JButton getButtonDelete() {
		return (JButton) getButton("ButtonDelete");
	}

	private JButton getButtonRefresh() {
		return (JButton) getButton("ButtonRefresh");
	}

	private JTable getTableAddresses() {
		return getTable("TableAddresses");
	}

	/**
	 * Sets the actual address controller implementation.
	 * 
	 * @param addressListController
	 *            Address controller.
	 */
	public final void setAddressController(
			final AddressController addressListController) {
		this.addressController = addressListController;
	}

	private Address getSelectedAddress() {
		final List<Address> selectedAddresses = addressSelection.getSelected();
		if (selectedAddresses.size() == 0) {
			return null;
		}
		return selectedAddresses.get(0);
	}

	private void setAdresses(final List<Address> list) {
		addresses.clear();
		addresses.addAll(list);
	}

	private int indexOf(final int id) {
		if (id < 0) {
			return -1;
		}
		for (int i = 0; i < addresses.size(); i++) {
			final Address address = addresses.get(i);
			if (address.getId() == id) {
				return i;
			}
		}
		return -1;
	}

	private void selectAdress(final int id) {
		if (id > 0) {
			return;
		}
		final int i = indexOf(id);
		if (i > -1) {
			getTableAddresses().getSelectionModel().setSelectionInterval(i, i);
		}
	}

	private void deleteAddress(final int id) {
		final int idx = indexOf(id);
		if (idx > -1) {
			addresses.remove(idx);
			if (addresses.size() == 0) {
				// Gets not repainted in this case... Seems to be a bug I
				// haven't investigated yet...
				repaint();
			} else {
				int row = idx;
				if (row >= addresses.size()) {
					row = addresses.size() - 1;
				}
				getTableAddresses().getSelectionModel().setSelectionInterval(
						row, row);
			}
		}
	}

	private void showNotImplementedDialog() {
		JOptionPane
				.showMessageDialog(
						AddressListPanel.this,
						"Sorry, not implemented yet!\n\n Only 'Refresh' and 'Delete' are available!\n\n",
						"Message", JOptionPane.INFORMATION_MESSAGE);
	}

}
