/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.examples.apps4swing;

import java.io.File;
import java.util.Properties;

import org.apache.log4j.PropertyConfigurator;
import org.fuin.apps4j.base.CanceledException;
import org.fuin.apps4swing.JFrameWindowManager;
import org.fuin.apps4swing.ManualMvcContext;
import org.fuin.apps4swing.MvcBundle;
import org.fuin.examples.apps4xxxxx.LoginController;
import org.fuin.examples.apps4xxxxx.LoginControllerImpl;
import org.fuin.examples.apps4xxxxx.LoginModule;
import org.fuin.examples.apps4xxxxx.LoginView;
import org.fuin.utils4swing.common.Utils4Swing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Better approach using an easy framework to connect controller and view.
 */
public final class LoginDialogBetterApproach {

    private static final Logger LOG = LoggerFactory.getLogger(LoginDialogBetterApproach.class);

    /**
     * Start the application in calling (main) thread.
     */
    public final void start() {

        LOG.info("start()");

        try {

            // Initialize Look and Feel
            Utils4Swing.initSystemLookAndFeel();

            // Create view (=panel)
            final LoginPanel view = new LoginPanel();

            // Create controller
            final LoginControllerImpl ctrl = new LoginControllerImpl();

            // Create MVC bundle and context
            final MvcBundle<LoginView, LoginController, LoginModule> mvcBundle = createMvcBundle(
                    view, ctrl);
            final ManualMvcContext mvcContext = createMvcContext(mvcBundle, ctrl, view);

            // Start the module in the main thread
            final LoginModule module = (LoginModule) mvcContext.getModuleImpl(mvcBundle);
            try {
                final String username = module.login("john");
                LOG.info("Login successful: " + username);
            } catch (final CanceledException ex) {
                LOG.info(ex.getMessage());
                LOG.info("exit(1)");
                System.exit(1);
            }

            // Finished
            LOG.info("exit(0)");
            System.exit(0);
        } catch (final Throwable t) {
            LOG.error("exit(2)", t);
            System.exit(2);
        }

    }

    /**
     * Create a manual context an initialize it.
     * 
     * @param mvcBundle
     *            Bundle to use.
     * @param ctrl
     *            Real controller implementation.
     * @param view
     *            Real view implementation.
     * 
     * @return New context.
     */
    private ManualMvcContext createMvcContext(
            final MvcBundle<LoginView, LoginController, LoginModule> mvcBundle,
            final LoginControllerImpl ctrl, final LoginPanel view) {

        // Create the context and initialize it
        final ManualMvcContext mvcContext = new ManualMvcContext();
        mvcContext.setBundles(mvcBundle);
        mvcContext.init();

        // Wire the two implementation classes manually together -
        // This could (and should) be done with Spring or other DI framework.
        ctrl.setWindowManager(new JFrameWindowManager());
        ctrl.setView((LoginView) mvcContext.getToViewDispatcher(mvcBundle));
        view.setController((LoginController) mvcContext.getToControllerDispatcher(mvcBundle));

        return mvcContext;
    }

    /**
     * Create an MVC bundle.
     * 
     * @param view
     *            Real view implementation.
     * @param ctrl
     *            Real controller implementation.
     * 
     * @return New bundle.
     */
    private MvcBundle<LoginView, LoginController, LoginModule> createMvcBundle(
            final LoginPanel view, final LoginControllerImpl ctrl) {

        final MvcBundle<LoginView, LoginController, LoginModule> mvcBundle;
        mvcBundle = new MvcBundle<LoginView, LoginController, LoginModule>();
        mvcBundle.setName("Login");
        mvcBundle.setPkg("org.fuin.examples.apps4swing.generated");
        mvcBundle.setCtrlIntf(LoginController.class);
        mvcBundle.setViewIntf(LoginView.class);
        mvcBundle.setModuleIntf(LoginModule.class);
        mvcBundle.setCtrlImpl(ctrl);
        mvcBundle.setViewImpl(view);

        // Write sources to a directory for debugging
        mvcBundle.setSrcDir(new File("C:/tmp/src-gen"));

        return mvcBundle;

    }

    private static void configureLog4J() {
        final Properties props = new Properties();
        props.put("log4j.rootLogger", "INFO, A1");
        props.put("log4j.appender.A1", "org.apache.log4j.ConsoleAppender");
        props.put("log4j.appender.A1.layout", "org.apache.log4j.PatternLayout");
        props.put("log4j.appender.A1.layout.ConversionPattern",
                "%d{HH:mm:ss,SSS} [%-20t] %-5p %-40c{1} - %m%n");
        PropertyConfigurator.configure(props);
    }

    /**
     * Starts the example.
     * 
     * @param args
     *            Not used.
     */
    public static void main(final String[] args) {
        configureLog4J();
        new LoginDialogBetterApproach().start();
    }

}
