/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.examples.jmsmvc4swing;

import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.fuin.aspects4swing.InvokeAndWait;
import org.fuin.aspects4swing.InvokeLater;
import org.fuin.jmsmvc4swing.base.ControllerPair;
import org.fuin.jmsmvc4swing.base.IControllerReceiver;
import org.fuin.jmsmvc4swing.base.JmsJndiEnvironment;
import org.fuin.jmsmvc4swing.base.JmsJndiEnvironmentImpl;
import org.fuin.jmsmvc4swing.model.ControllerModel;
import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.utils4swing.common.ScreenCenterPositioner;
import org.fuin.utils4swing.common.Utils4Swing;

/**
 * Java Swing based address manager.
 */
public class AddressManager {

	private final JmsJndiEnvironment env;

	private AddressController sender;

	private IControllerReceiver receiver;

	private final ControllerModel<AddressController> controllerModel;

	static {
		// Ugly workaround to enable application for Webstart
		// 
		// TODO Investigate why Webstart throws exception even if we have
		// "all-permissions" included in JNLP! Exception is as follows:
		// java.security.AccessControlException: access denied
		// (javax.management.MBeanServerPermission createMBeanServer)
		System.setSecurityManager(null);
	}

	/**
	 * Constructor with environment.
	 * 
	 * @param env
	 *            Environment.
	 */
	public AddressManager(final JmsJndiEnvironment env) {
		super();
		if (env == null) {
			throw new IllegalArgumentException(
					"The argument 'env' cannot be null!");
		}
		this.env = env;

		// We need a "class pool" and a byte code generator with a special
		// classloader here.
		// Without this Webstart fails with Javassist generated classes!
		this.controllerModel = new ControllerModel<AddressController>(
				"org.fuin.jaddr.onthefly", AddressController.class,
				ByteCodeGenerator.createWithCurrentThreadContextClassLoader());
		init();
	}

	private static void registerShutdownHook(final JmsJndiEnvironment env) {
		final Runnable runnable = new Runnable() {
			public void run() {
				env.close();
			}
		};
		Runtime.getRuntime().addShutdownHook(new Thread(runnable));
	}

	/**
	 * Show the UI.
	 */
	@InvokeLater
	public final void execute() {
		final AddressListPanel panel = new AddressListPanel(controllerModel
				.getPool(), controllerModel.getGenerator());
		panel.setAddressController(sender);
		Utils4Swing.createShowAndPosition("JAddressManager", panel.getView(),
				true, true, new ScreenCenterPositioner());
	}

	private void init() {
		// Topic used for controller and view
		final String topicName = AddressController.class.getName();

		final ControllerPair<AddressController> result = controllerModel
				.createOnTheFly(env, topicName, new AddressControllerImpl());

		// Receiver for the controller
		receiver = result.getReceiver();

		System.out.println("RECEIVER=" + receiver);
		receiver.start();

		// Sender for the view
		sender = result.getSender();
		System.out.println("SENDER=" + sender);

	}

	@InvokeAndWait
	private static void initLookAndFeel() {
		Utils4Swing.initSystemLookAndFeel();
	}

	/**
	 * Start the application.
	 * 
	 * @param args
	 *            Command line arguments.
	 * 
	 * @throws NamingException
	 *             Problem with the JNDI context.
	 */
	public static void main(final String[] args) throws NamingException {

		final Properties props = new Properties();
		props.put(Context.INITIAL_CONTEXT_FACTORY,
				"net.walend.somnifugi.SomniTopicContextFactory");
		final String topicConnectionFactoryKey = "ConnectionFactory";

		// --- Example with "ActiveMQ" JMS server ---
		// props.put(Context.INITIAL_CONTEXT_FACTORY,
		// "org.apache.activemq.jndi.ActiveMQInitialContextFactory");
		// props.put(Context.PROVIDER_URL, "tcp://localhost:61616");
		// final String topicConnectionFactoryKey = "TopicConnectionFactory";
		// ------------------------------------------

		final JmsJndiEnvironment env = new JmsJndiEnvironmentImpl(
				new InitialContext(props), topicConnectionFactoryKey);
		env.open();
		registerShutdownHook(env);
		initLookAndFeel();
		new AddressManager(env).execute();

	}

}
