/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.examples.apps4xxxxx;

import java.util.Arrays;

import org.fuin.apps4j.CanceledException;
import org.fuin.apps4j.WindowManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Example login module and controller implementation.
 */
public class LoginControllerImpl implements LoginController, LoginModuleImplIntf {

    private static final Logger LOG = LoggerFactory.getLogger(LoginControllerImpl.class);

    private LoginView view;

    private LoginListener listener;

    private WindowManager windowManager;

    /**
     * Default constructor.
     */
    public LoginControllerImpl() {
        super();
    }

    /**
     * Current thread sleeps for some time.
     * 
     * @param millis
     *            Milliseconds.
     */
    private void sleep(final long millis) {
        try {
            Thread.sleep(millis);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    /**
     * Checks if the password is "test".
     * 
     * @param input
     *            Password to check.
     * 
     * @return If the password was "test" <code>true</code> else
     *         <code>false</code>.
     */
    private boolean isPasswordCorrect(final char[] input) {

        // Simulate long user/pw check
        sleep(5000);

        boolean isCorrect = true;
        final char[] correctPassword = { 't', 'e', 's', 't' };

        if (input.length != correctPassword.length) {
            isCorrect = false;
        } else {
            isCorrect = Arrays.equals(input, correctPassword);
        }
        // Zero out the password.
        Arrays.fill(correctPassword, '0');
        return isCorrect;
    }

    /**
     * {@inheritDoc}
     */
    public final void setView(final LoginView view) {
        this.view = view;
    }

    /**
     * {@inheritDoc}
     */
    public final void cancel() {
        LOG.info("cancel");

        // Hide the UI
        windowManager.close();

        if (listener != null) {
            // Notify the listener that the login was canceled
            listener.failure(new CanceledException());
            listener = null;
        }

    }

    /**
     * {@inheritDoc}
     */
    public final void verify(final String username, final char[] password) {
        LOG.info("verify '" + username + "'");

        if (isPasswordCorrect(password)) {
            LOG.info("LOGIN OK!");

            // Hide the UI
            windowManager.close();

            // Notify the listener that the login was successful
            listener.success(username);
            listener = null;

        } else {
            final String message = "INVALID USERNAME OR PASSWORD";
            LOG.info(message);
            view.setMessage(message);
        }

    }

    /**
     * {@inheritDoc}
     */
    public final void login(final LoginListener listener, final String username) {
        LOG.info("login '" + username + "'");

        // Show the UI
        windowManager.open();

        // Set values in view
        this.view.setUsername(username);
        this.view.setMessage("");

        // Keep the listener to inform on "cancel()" or "verify(..)"
        this.listener = listener;

    }

    /**
     * {@inheritDoc}
     */
    public final void login(final LoginListener listener) {
        LOG.info("login");
        
        // Show the UI
        windowManager.open();
        
        // Set values in view
        this.view.setUsername("");
        this.view.setMessage("");

        // Keep the listener to inform on "cancel()" or "verify(..)"
        this.listener = listener;
        
    }

    /**
     * {@inheritDoc}
     */
    public final String getModuleTitle() {
        return "Login Example Dialog";
    }

    /**
     * {@inheritDoc}
     */
    public final Object getModuleUI() {
        return view.getViewUI();
    }

    /**
     * {@inheritDoc}
     */
    public final void setWindowManager(final WindowManager windowManager) {
        this.windowManager = windowManager;
    }

}
