/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.examples.srcgen4javassist;

import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;

/**
 * Simple example for generating source code.
 */
public final class GenerateExample {

    /**
     * Private constructor.
     */
    private GenerateExample() {
        throw new UnsupportedOperationException(
                "It's not allowed to create an instance of this class!");
    }

    /**
     * Creates the model class.
     * 
     * @param pool
     *            Pool.
     * @param packageName
     *            Package of the class.
     * 
     * @return Model class.
     */
    public static SgClass create(final SgClassPool pool, final String packageName) {

        // Create class
        final SgClass clasz = new SgClass(packageName, "MyImpl");
        clasz.addInterface(SgClass.create(pool, MyIntf.class));

        // Create fields - This implicitly adds the field to the class
        new SgField(clasz, "public", SgClass.create(pool, String.class), "name", "\"unknown\"");

        // Create first method - This implicitly adds the method to the class
        final SgMethod getNameMethod = new SgMethod(clasz, "public", SgClass.create(pool,
                String.class), "getName");
        getNameMethod.addBodyLine("return name;");

        // Create second method - This implicitly adds the method to the class
        final SgMethod setNameMethod = new SgMethod(clasz, "public", SgClass.VOID, "setName");
        // Create argument - This implicitly adds the argument to the method
        new SgArgument(setNameMethod, SgClass.create(pool, String.class), "name");
        setNameMethod.addBodyLine("this.name = name;");

        // Create third method - This implicitly adds the method to the class
        final SgMethod sayHelloMethod = new SgMethod(clasz, "public", SgClass.VOID, "sayHello");
        sayHelloMethod.addBodyLine("System.out.println(\"Hello \" + name + \"!\");");

        return clasz;
    }

    /**
     * Starts the source code generation.
     * 
     * @param args
     *            Not used.
     */
    public static void main(final String[] args) {

        // Create a cache for generated classes
        final SgClassPool pool = new SgClassPool();

        // Create the class
        final SgClass clasz = create(pool, "test");

        // Print the source
        System.out.println(clasz);
        System.out.println("-----------");

        // Create an instance
        final MyIntf example = (MyIntf) (new ByteCodeGenerator()).createInstance(clasz);
        example.setName("John");
        example.sayHello();

    }

}
