/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.examples.kickstart4j;

import java.io.File;
import java.io.IOException;

import org.fuin.kickstart4j.Config;
import org.fuin.kickstart4j.ConfigParser;
import org.fuin.kickstart4j.InvalidConfigException;
import org.fuin.kickstart4j.MkDir;
import org.fuin.kickstart4j.SrcFile;
import org.fuin.utils4j.Utils4J;

/**
 * Example for creating a XML configuration file for Kickstart4J.
 */
public final class CreateConfigExample {

	private static final boolean UNZIP = true;
	private static final boolean DONT_UNZIP = false;
	private static final boolean LOAD_ALWAYS = true;
	private static final boolean ADD_TO_CLASSPATH = true;
	private static final boolean DONT_ADD_TO_CLASSPATH = false;

	/**
	 * Private constructor.
	 */
	private CreateConfigExample() {
		throw new UnsupportedOperationException(
				"It's not allowed to create an instance of this class!");
	}

	/**
	 * Creates configuration file.
	 * 
	 * @param args
	 *            Not used.
	 * 
	 * @throws IOException
	 *             Error writing XML configuration file.
	 * @throws InvalidConfigException
	 *             Error reading an existing configuration.
	 */
	@SuppressWarnings("unchecked")
	public static void main(final String[] args) throws IOException,
			InvalidConfigException {

		final String url = "http://www.fuin.org/examples/kickstart4j";
		final File configFile = new File("src/main/kickstart4j/example.xml");

		// Read the existing values (if the file exists)
		final Config config = ConfigParser.create(configFile);

		// Set basic information
		config.setIdFilename(".org-fuin-examples-SwingSet2");
		config.setDestPath("c:/fuin.org/examples/swingset2");
		config.setJavaExe("jre6/bin/java.exe");
		config.setJavaArgs("-classpath ${classpath} SwingSet2");
		config.setTitle("Kickstart4J SwingSet2");
		config.setVendor("Future Invent Informationsmanagement GmbH");
		config.setDescription("Local installed SwingSet2 with "
				+ "it's own Java Runtime 1.6");
		config.setLogFilename("${destDir}/logs/kickstart.log");
		config.setVersion("0.1.0");

		// Remove all existing file/directory definitions before adding new ones
		config.getSrcFiles().clear();
		config.getSrcDirs().clear();
		config.getMkDirs().clear();
		
		// Add the main application jar file 
		final File swingSet2Jar = new File("src/main/kickstart4j/SwingSet2.jar");
		final SrcFile swingSet2SrcFile = new SrcFile("lib", "SwingSet2.jar",
				Utils4J.createHash(swingSet2Jar), swingSet2Jar.length(),
				DONT_UNZIP, LOAD_ALWAYS, ADD_TO_CLASSPATH, url
						+ "/SwingSet2.jar");
		config.getSrcFiles().add(swingSet2SrcFile);

		// Add the JRE zip file
		final File jre6Zip = new File("src/main/kickstart4j/jre6.zip");
		final SrcFile jre6SrcFile = new SrcFile("", "jre6.zip", Utils4J
				.createHash(jre6Zip), jre6Zip.length(), UNZIP, LOAD_ALWAYS,
				DONT_ADD_TO_CLASSPATH, url + "/jre6.zip");
		config.getSrcFiles().add(jre6SrcFile);
		
		// Create an empty log directory at the target
		config.getMkDirs().add(new MkDir("logs"));

		// Write the config to disk
		config.writeToVarXML(configFile, false);

		System.out.println("CREATED " + configFile.getCanonicalPath());

	}

}
