/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.examples.jmsmvc4swing;

import java.io.Serializable;

import net.sf.oval.constraint.Length;
import net.sf.oval.constraint.NotNegative;
import net.sf.oval.constraint.NotNull;

import org.fuin.utils4swing.annotations.Label;
import org.fuin.utils4swing.annotations.TableColumn;

/**
 * Address information. 
 */
@Label("Address")
public final class Address implements Serializable, Comparable<Address> {

    private static final long serialVersionUID = 1L;

    @NotNegative
    @Label("Unique ID")
    private int id = 0;

    @NotNull
    @Length(max = 40)
    @Label("Last name")
    @TableColumn(width = 100, pos = 0)
    private String lastName = "";

    @NotNull
    @Length(max = 40)
    @Label("First name")
    @TableColumn(width = 100, pos = 1)
    private String firstName = "";

    @NotNull
    @Length(max = 40)
    @Label("Street")
    @TableColumn(width = 130, pos = 2)
    private String street = "";

    @NotNull
    @Length(max = 10)
    @Label("ZIP-Code")
    @TableColumn(width = 50, pos = 3)
    private String zip = "";

    @NotNull
    @Length(max = 30)
    @Label("City")
    @TableColumn(width = 50, pos = 4)
    private String city = "";

    @NotNull
    @Length(max = 2)
    @Label("State")
    @TableColumn(width = 60, pos = 5)
    private String state = "";

    @NotNull
    @Length(max = 2)
    @Label("Country")
    @TableColumn(width = 60, pos = 6)
    private String country = "";

    /**
     * Default constructor.
     */
    public Address() {
        super();
    }

    /**
     * Returns the unique identifier.
     * 
     * @return Unique ID.
     */
    public final int getId() {
        return id;
    }

    /**
     * Sets the unique identifier to a new value.
     * 
     * @param id Unique ID.
     * 
     * @return The address instance.
     */
    public final Address setId(final int id) {
        this.id = id;
        return this;
    }

    /**
     * Returns the last name.
     * 
     * @return Last name.
     */
    public final String getLastName() {
        return lastName;
    }

    /**
     * Sets the last name to a new value.
     * 
     * @param lastName Last name.
     * 
     * @return The address instance.
     */
    public final Address setLastName(final String lastName) {
        this.lastName = lastName;
        return this;
    }

    /**
     * Returns the first name.
     * 
     * @return First name.
     */
    public final String getFirstName() {
        return firstName;
    }

    /**
     * Sets the first name to a new value.
     * 
     * @param firstName Last name.
     * 
     * @return The address instance.
     */
    public final Address setFirstName(final String firstName) {
        this.firstName = firstName;
        return this;
    }

    /**
     * Returns the street.
     * 
     * @return Name of the street.
     */
    public final String getStreet() {
        return street;
    }

    /**
     * Sets the street name to a new value.
     * 
     * @param street Name of the street.
     * 
     * @return The address instance.
     */
    public final Address setStreet(final String street) {
        this.street = street;
        return this;
    }

    /**
     * Returns the postal ZIP code.
     * 
     * @return ZIP code.
     */
    public final String getZip() {
        return zip;
    }

    /**
     * Sets the postal ZIP code to a new value.
     * 
     * @param zip ZIP code.
     * 
     * @return The address instance.
     */
    public final Address setZip(final String zip) {
        this.zip = zip;
        return this;
    }

    /**
     * Returns the city.
     * 
     * @return Name of the city.
     */
    public final String getCity() {
        return city;
    }

    /**
     * Sets the city to a new value.
     * 
     * @param city Name of the city.
     * 
     * @return The address instance.
     */
    public final Address setCity(final String city) {
        this.city = city;
        return this;
    }

    /**
     * Returns the state.
     * 
     * @return Name of the state.
     */
    public final String getState() {
        return state;
    }

    /**
     * Sets the state name to a new value.
     * 
     * @param state Name of the state.
     * 
     * @return The address instance.
     */
    public final Address setState(final String state) {
        this.state = state;
        return this;
    }

    /**
     * Returns the country.
     * 
     * @return Name of the country.
     */
    public final String getCountry() {
        return country;
    }

    /**
     * Sets the country to a new value.
     * 
     * @param country Name of the country.
     * 
     * @return The address instance.
     */
    public final Address setCountry(final String country) {
        this.country = country;
        return this;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + id;
        return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Address other = (Address) obj;
        if (id != other.id) {
            return false;
        }
        return true;
    }

    /**
     * {@inheritDoc}
     */
    public final int compareTo(final Address other) {
        if (id > other.id) {
            return 1;
        } else if (id < other.id) {
            return -1;
        }
        return 0;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        return "id=" + id + ", " + "lastName=" + lastName + ", " + "firstName=" + firstName
                + ", " + "street=" + street + ", " + "zip=" + zip + ", " + "city=" + city
                + ", " + "state=" + state + ", " + "country=" + country;
    }

}
