package org.fuin.aspects4swing;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.reflect.CodeSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Aspect for tracing entering and exiting methods. Inspired by examples from
 * the <a
 * href="http://www.eclipse.org/aspectj/doc/released/progguide/index.html">The
 * AspectJ (TM) Programming Guide</a>.
 */
public abstract aspect Sl4jTrace {

    protected static int callDepth = 0;

    private static String spaces = "  ";
    
    /**
     * Define indent spaces to use.
     * 
     * @param newSpaces Indent spaces.
     */
    public static void setIndent(final String newSpaces) {
        if (newSpaces == null) {
            spaces = "";
        } else {
            spaces = newSpaces;
        }
    }
    
    /**
     * Logs entering a method.
     * 
     * @param log
     *            Logger to use.
     * @param jp
     *            Current join point.
     */
    protected static void logBegin(final Logger log, final JoinPoint jp) {
        callDepth++;
        final String indent = getIndent();
        log.trace(indent + "BEGIN " + jp.getSignature().getName() + "(" + getArgs(jp) + ")");
        logArguments(log, indent, jp);
    }

    /**
     * Logs returning from a function.
     * 
     * @param log
     *            Logger to use.
     * @param jp
     *            Current join point.
     * @param result
     *            Result of the function call.
     */
    protected static void logFunctionEnd(final Logger log, final JoinPoint jp, final Object result) {
        final String indent = getIndent();
        log.trace(indent + "result=" + result);
        log.trace(indent + "END " + jp.getSignature().getName());
        callDepth--;
    }

    /**
     * Logs returning from a method.
     * 
     * @param log
     *            Logger to use.
     * @param jp
     *            Current join point.
     */
    protected static void logEnd(final Logger log, final JoinPoint jp) {
        log.trace(getIndent() + "END " + jp.getSignature().getName());
        callDepth--;
    }

    /**
     * Calculates the spaces depending on the call stack.
     * 
     * @return Spaces.
     */
    private static String getIndent() {
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < callDepth; i++) {
            sb.append(spaces);
        }
        return sb.toString();
    }

    /**
     * Returns all argument types as String
     * 
     * @param jp
     *            Current join point.
     * 
     * @return Argument type names separated by a comma.
     */
    @SuppressWarnings("unchecked")
    private static String getArgs(final JoinPoint jp) {
        final Class[] types = ((CodeSignature) jp.getSignature()).getParameterTypes();
        final StringBuffer sb = new StringBuffer();
        for (int i = 0; i < types.length; i++) {
            if (i > 0) {
                sb.append(", ");
            }
            sb.append(types[i].getSimpleName());
        }
        return sb.toString();
    }

    /**
     * Logs all arguments as key-value pair (one per line).
     * 
     * @param log
     *            Logger to use.
     * @param jp
     *            Current join point.
     * @param indent
     *            Leading spaces.
     */
    private static void logArguments(final Logger log, final String indent, final JoinPoint jp) {
        final Object[] args = jp.getArgs();
        final String[] names = ((CodeSignature) jp.getSignature()).getParameterNames();
        for (int i = 0; i < args.length; i++) {
            log.trace(indent + names[i] + "=" + args[i]);
        }
    }

    /**
     * Concrete "classes" are defined in the subclasses.
     */
    public abstract pointcut aClass();

    /**
     * All constructors.
     */
    pointcut aConstructor(): aClass() && execution(new(..));

    /**
     * All methods returning a result except "toString()".
     */
    pointcut aFunction(): aClass() && execution(* *(..)) && !aMethod() && !execution(String toString());

    /**
     * All methods without a result.
     */
    pointcut aMethod(): aClass() && execution(void *(..));

    /**
     * Before calling a constructor.
     */
    before(): aConstructor() {
        final Logger log = LoggerFactory.getLogger(thisJoinPointStaticPart.getSignature()
                .getDeclaringType());
        if (log.isTraceEnabled()) {
            logBegin(log, thisJoinPoint);
        }
    }

    /**
     * After calling a constructor.
     */
    after(): aConstructor() {
        final Logger log = LoggerFactory.getLogger(thisJoinPointStaticPart.getSignature()
                .getDeclaringType());
        if (log.isTraceEnabled()) {
            logEnd(log, thisJoinPoint);
        }
    }

    /**
     * Before calling a method (with void result)
     */
    before(): aMethod() {
        final Logger log = LoggerFactory.getLogger(thisJoinPointStaticPart.getSignature()
                .getDeclaringType());
        if (log.isTraceEnabled()) {
            logBegin(log, thisJoinPoint);
        }
    }

    /**
     * After calling a method (with void result).
     */
    after(): aMethod() {
        final Logger log = LoggerFactory.getLogger(thisJoinPointStaticPart.getSignature()
                .getDeclaringType());
        if (log.isTraceEnabled()) {
            logEnd(log, thisJoinPoint);
        }
    }

    /**
     * Calling a method with result.
     */
    Object around(): aFunction() && !execution(* aFunction()) {
        final Logger log = LoggerFactory.getLogger(thisJoinPointStaticPart.getSignature()
                .getDeclaringType());
        if (log.isTraceEnabled()) {
            logBegin(log, thisJoinPoint);
            Object result = proceed();
            logFunctionEnd(log, thisJoinPoint, result);
            return result;
        } else {
            return proceed();
        }
    }

}
