/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4swing;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgUtils;
import org.fuin.srcgen4javassist.SgVariable;
import org.fuin.srcgen4javassist.factory.ImplementationFactoryListener;
import org.fuin.srcgen4javassist.factory.VarListImplementationFactory;
import org.fuin.utils4j.ToDebugStringCapable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility methods for the package.
 */
public final class Utils {
    
    private static final Logger LOG = LoggerFactory.getLogger(Utils.class);

    /**
     * Private constructor.
     */
    private Utils() {
        throw new UnsupportedOperationException(
                "Creating an instance of a utility class is not allowed!");
    }

    /**
     * Adds a private static LOG variable to the class.
     * 
     * @param pool
     *            Class pool.
     * @param clasz
     *            Class to add a Logger to.
     */
    public static void addPrivateStaticLogger(final SgClassPool pool, final SgClass clasz) {
        final SgClass sl4jLoggerClasz = SgClass.create(pool, Logger.class);
        clasz.addField(new SgField(clasz, "private static", sl4jLoggerClasz, "LOG",
                LoggerFactory.class.getName() + ".getLogger(" + clasz.getName() + ".class)"));
    }

    /**
     * Creates a trace line for a variable.
     * 
     * @param pool
     *            Class pool.
     * @param lines
     *            The statement is added to this collection.
     * @param varName
     *            Name of the variable.
     * @param varType
     *            Type of the variable.
     */
    public static void addVarTraceStmt(final SgClassPool pool, final List<String> lines,
            final String varName, final SgClass varType) {
        if (varType.isPrimitive() || varType.isBaseType()) {
            lines.add("LOG.trace(\"" + varName + "=\" + " + varName + ");");
        } else {
            if (varType.hasInterface(SgClass.create(pool, ToDebugStringCapable.class))) {
                lines.add("LOG.trace(\"" + varName + "=\" + (("
                        + ToDebugStringCapable.class.getName() + ") " + varName + ")"
                        + ".toDebugString());");
            } else {
                lines.add("LOG.trace(\"" + varName + "=\" + " + varName + ");");
            }
        }
    }

    /**
     * Add some log lines for a method call including method name and arguments
     * (DEBUG) and argument values (TRACE).
     * 
     * @param pool
     *            Class pool.
     * @param lines
     *            The statement is added to this collection.
     * @param method
     *            Method to create a call trace for.
     */
    public static void addMethodCallLogStmt(final SgClassPool pool, final List<String> lines,
            final SgMethod method) {
        lines.add("if (LOG.isDebugEnabled()) {");
        lines.add("    LOG.debug(\"" + method.getTypeSignature() + "\");");
        lines.add("}");
        final List<SgArgument> args = method.getArguments();
        if (args.size() > 0) {
            lines.add("if (LOG.isTraceEnabled()) {");
            for (int i = 0; i < args.size(); i++) {
                final SgArgument arg = args.get(i);
                addVarTraceStmt(pool, lines, arg.getName(), arg.getType());
            }
            lines.add("}");
        }
    }

    /**
     * Creates a class that implements {@link Runnable} that calls a given
     * method. The implementation is created inside the same package as the
     * class that owns the <code>method</code> that is called.
     * 
     * @param pool
     *            Class pool.
     * @param generator
     *            Byte code generator.
     * @param varListFactory
     *            Helper factory.
     * @param var
     *            Name and type of the variable.
     * @param method
     *            Method to call on the variable in the <code>run()</code> body.
     * 
     * @return Runnable implementation.
     */
    public static SgClass createRunnableClass(final SgClassPool pool,
            final ByteCodeGenerator generator, final VarListImplementationFactory varListFactory,
            final SgVariable var, final SgMethod method) {

        final List<SgVariable> runnableArgs = new ArrayList<SgVariable>();
        runnableArgs.add(var);
        runnableArgs.addAll(method.getArguments());
        final String implPackage = method.getOwner().getPackageName();
        final String implName = "Runnable" + SgUtils.firstCharUpper(method.getName());
        final SgClass runnableClass = varListFactory.create(implPackage, implName, null, method
                .getOwner(), runnableArgs, new ImplementationFactoryListener() {
            public void afterClassCreated(final SgClass clasz) {
                // Do nothing
            }

            public List<String> createBody(final SgMethod runMethod, final Class<?>... intf) {
                final List<String> lines = new ArrayList<String>();
                lines.add(var.getName() + "." + method.getCallSignature() + ";");
                return lines;
            }
        }, Runnable.class);
        
        if (LOG.isTraceEnabled()) {
            LOG.trace("Created Runnable\n" + runnableClass);
        }

        try {
            generator.createClass(runnableClass);
        } catch (final Throwable throwable) {
            final String msg = "Error creating Runnable";
            LOG.error(msg + ":\n" + runnableClass);
            throw new RuntimeException(msg, throwable);
        }
        return runnableClass;
    }

    /**
     * Creates a class that implements {@link EDTGetter} that calls a given
     * method. The implementation is created inside the same package as the
     * class that owns the <code>method</code> that is called.
     * 
     * @param pool
     *            Class pool.
     * @param generator
     *            Byte code generator.
     * @param varListFactory
     *            Helper factory.
     * @param var
     *            Name and type of the variable.
     * @param method
     *            Method to call on the variable in the <code>get()</code> body.
     * 
     * @return Runnable implementation.
     */
    public static SgClass createEDTGetterClass(final SgClassPool pool,
            final ByteCodeGenerator generator, final VarListImplementationFactory varListFactory,
            final SgVariable var, final SgMethod method) {

        final List<SgVariable> getterArgs = new ArrayList<SgVariable>();
        getterArgs.add(var);
        getterArgs.addAll(method.getArguments());

        final String implPackage = method.getOwner().getPackageName();
        final String implName = "EDTGetter" + SgUtils.firstCharUpper(method.getName());
        final SgClass getterClass = varListFactory.create(implPackage, implName, null, method
                .getOwner(), getterArgs, new ImplementationFactoryListener() {
            public void afterClassCreated(final SgClass clasz) {
                // Do nothing
            }

            public List<String> createBody(final SgMethod runMethod, final Class<?>... intf) {
                final List<String> lines = new ArrayList<String>();
                lines.add("return " + var.getName() + "." + method.getCallSignature() + ";");
                return lines;
            }
        }, EDTGetter.class);

        generator.createClass(getterClass);
        return getterClass;
    }

    public static void writeToFile(final File file, String text) {
        try {
            final FileWriter writer = new FileWriter(file);
            try {
                writer.write(text);
            } finally {
                writer.close();
            }
        } catch (final IOException ex) {
            throw new RuntimeException(ex);
        }
    }

}
