/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4swing;

import java.util.ArrayList;
import java.util.List;

import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgVariable;
import org.fuin.srcgen4javassist.factory.ImplementationFactoryListener;
import org.fuin.srcgen4javassist.factory.VarListImplementationFactory;
import org.fuin.utils4j.Utils4J;
import org.fuin.utils4swing.annotations.InvokeAndWait;

/**
 * Creates the method bodies for a {@link AbstractToViewDispatcher} implementation.
 */
public class ToViewDispatcherFactoryListener implements ImplementationFactoryListener {

	private final SgClassPool pool;

	private final ByteCodeGenerator generator;

	private final VarListImplementationFactory varListFactory;

	/**
	 * Constructor with pool, generator and interface.
	 * 
	 * @param pool
	 *            Class pool.
	 * @param generator
	 *            Byte code generator.
	 */
	public ToViewDispatcherFactoryListener(final SgClassPool pool,
			final ByteCodeGenerator generator) {
		super();

		Utils4J.checkNotNull("pool", pool);
		this.pool = pool;

		Utils4J.checkNotNull("generator", generator);
		this.generator = generator;

		this.varListFactory = new VarListImplementationFactory(pool);
	}

	/**
	 * {@inheritDoc}
	 */
	public final void afterClassCreated(final SgClass clasz) {
		Utils.addPrivateStaticLogger(pool, clasz);
	}

	/**
	 * {@inheritDoc}
	 */
	public final List<String> createBody(final SgMethod method, final Class<?>... intf) {
		final List<String> lines = new ArrayList<String>();

		Utils.addMethodCallLogStmt(pool, lines, method);

		if (method.getReturnType().equals(SgClass.VOID)) {
			final SgVariable var = new SgVariable("", SgClass.create(pool, intf[0]), "view") {
			};
			final SgClass runnableClass = Utils.createRunnableClass(pool, generator,
					varListFactory, var, method);
			lines.add(intf[0].getName() + " view = ((" + intf[0].getName() + ") getView());");
			if (method.getArguments().size() == 0) {
				lines.add("Runnable runnable = new " + runnableClass.getSourceName() + "(view);");
			} else {
				lines.add("Runnable runnable = new " + runnableClass.getSourceName() + "(view, "
						+ method.getCommaSeparatedArgumentNames() + ");");
			}
			if (method.hasAnnotation(InvokeAndWait.class.getName())) {
				lines.add(EDTInvoker.class.getName() + ".invokeAndWait(runnable);");
			} else {
				lines.add(EDTInvoker.class.getName() + ".invokeLater(runnable);");
			}
		} else {
			final SgVariable var = new SgVariable("", SgClass.create(pool, intf[0]), "view") {
			};
			final SgClass edtGetterClass = Utils.createEDTGetterClass(pool, generator,
					varListFactory, var, method);
			lines.add(intf[0].getName() + " view = ((" + intf[0].getName() + ") getView());");
			if (method.getArguments().size() == 0) {
				lines.add(EDTGetter.class.getName() + " edtGetter = new "
						+ edtGetterClass.getSourceName() + "(view);");
			} else {
				lines.add(EDTGetter.class.getName() + " edtGetter = new "
						+ edtGetterClass.getSourceName() + "(view, "
						+ method.getCommaSeparatedArgumentNames() + ");");
			}
			if (method.getReturnType().isPrimitive()) {
				final SgClass nonPrimitiveClass = SgClass.getNonPrimitiveClass(pool, method
						.getReturnType());
				final String convMethod = SgClass.getToPrimitiveMethod(nonPrimitiveClass);
				lines.add(method.getReturnType().getName() + " retVal = (("
						+ nonPrimitiveClass.getName() + ") new " + EDTInvoker.class.getName()
						+ "(edtGetter).invoke())." + convMethod + "();");
			} else {
				lines.add(method.getReturnType().getName() + " retVal = new "
						+ EDTInvoker.class.getName() + "(edtGetter).invoke();");
			}
			Utils.addVarTraceStmt(pool, lines, "retVal", method.getReturnType());
			lines.add("return retVal;");
		}
		return lines;
	}

}
