/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4swing;

import java.util.ArrayList;
import java.util.List;

import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgVariable;
import org.fuin.srcgen4javassist.factory.ImplementationFactoryListener;
import org.fuin.srcgen4javassist.factory.VarListImplementationFactory;
import org.fuin.utils4j.Utils4J;

/**
 * Creates the method bodies for a {@link AbstractToControllerDispatcher}
 * -Implementation.
 */
public class ToControllerDispatcherFactoryListener implements ImplementationFactoryListener {

	private final SgClassPool pool;

	private final ByteCodeGenerator generator;

	private final VarListImplementationFactory varListFactory;

	/**
	 * Constructor with pool, generator and controller interface.
	 * 
	 * @param pool
	 *            Class pool.
	 * @param generator
	 *            Byte code generator.
	 */
	public ToControllerDispatcherFactoryListener(final SgClassPool pool,
			final ByteCodeGenerator generator) {
		super();

		Utils4J.checkNotNull("pool", pool);
		this.pool = pool;

		Utils4J.checkNotNull("generator", generator);
		this.generator = generator;

		this.varListFactory = new VarListImplementationFactory(pool);
	}

	/**
	 * {@inheritDoc}
	 */
	public final void afterClassCreated(final SgClass clasz) {
		Utils.addPrivateStaticLogger(pool, clasz);
	}

	/**
	 * {@inheritDoc}
	 */
	public final List<String> createBody(final SgMethod method, final Class<?>... intf) {

		final List<String> lines = new ArrayList<String>();
		Utils.addMethodCallLogStmt(pool, lines, method);

		final SgVariable var = new SgVariable("", SgClass.create(pool, intf[0]), "ctrl") {
		};
		final SgClass runnableClass = Utils.createRunnableClass(pool, generator, varListFactory,
				var, method);
		lines.add(intf[0].getName() + " ctrl = ((" + intf[0].getName() + ") getController());");
		if (method.getArguments().size() == 0) {
			lines.add("Runnable runnable = new " + runnableClass.getSourceName() + "(ctrl);");
		} else {
			lines.add("Runnable runnable = new " + runnableClass.getSourceName() + "(ctrl, "
					+ method.getCommaSeparatedArgumentNames() + ");");
		}
		lines.add("execute(runnable);");

		return lines;
	}

}
