/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4swing;

import java.util.ArrayList;
import java.util.List;

import org.fuin.apps4j.ModulImplIntfRef;
import org.fuin.apps4j.Module;
import org.fuin.apps4j.ModuleImplIntf;
import org.fuin.apps4j.ModuleImplIntfListener;
import org.fuin.apps4j.WaitForUserInput;
import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgUtils;
import org.fuin.srcgen4javassist.factory.ImplementationFactory;
import org.fuin.srcgen4javassist.factory.ImplementationFactoryListener;
import org.fuin.utils4j.Utils4J;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Creates the method bodies for a {@link AbstractModuleImpl} implementation.
 */
public final class ModuleImplFactoryListener implements ImplementationFactoryListener {

    private static final Logger LOG = LoggerFactory.getLogger(ModuleImplFactoryListener.class);

    private ImplementationFactory factory;

    private ByteCodeGenerator generator;

    private final ModuleImplIntfListenerFactoryListener listenerFactoryListener;

    private final SgClass moduleImplIntfClass;

    private final SgClass moduleImplIntfListenerClass;

    private final SgClassPool pool;

    /**
     * Constructor with class pool and module implementation interface.
     * 
     * @param pool
     *            Pool - Cannot be <code>null</code>.
     * @param moduleImplIntf
     *            Module implementation interface - Cannot be <code>null</code>.
     */
    public ModuleImplFactoryListener(final SgClassPool pool,
            final Class<? extends ModuleImplIntf> moduleImplIntf) {
        super();

        Utils4J.checkNotNull("pool", pool);
        this.pool = pool;

        Utils4J.checkNotNull("moduleImplIntf", moduleImplIntf);
        moduleImplIntfClass = SgClass.create(pool, moduleImplIntf);
        if (LOG.isDebugEnabled()) {
            LOG.debug("moduleImplIntf=" + moduleImplIntf.getName());
        }

        factory = new ImplementationFactory(pool, true);
        generator = new ByteCodeGenerator();
        listenerFactoryListener = new ModuleImplIntfListenerFactoryListener();
        moduleImplIntfListenerClass = SgClass.create(pool, ModuleImplIntfListener.class);
    }

    /**
     * {@inheritDoc}
     */
    public final void afterClassCreated(final SgClass clasz) {

        final SgClass sl4jLoggerClasz = SgClass.create(pool, Logger.class);
        clasz.addField(new SgField(clasz, "private static", sl4jLoggerClasz, "LOG",
                "org.slf4j.LoggerFactory.getLogger(" + clasz.getName() + ".class)"));

    }

    private boolean isSameArgumentsPlusListener(final SgMethod moduleIntfMethod,
            final SgMethod moduleImplIntfMethod) {
        if (moduleIntfMethod.getArguments().size() != moduleImplIntfMethod.getArguments().size() + 1) {
            return false;
        }
        final List<SgArgument> moduleIntfMethodArgs = moduleIntfMethod.getArguments();
        final List<SgArgument> moduleImplIntfMethodArgs = moduleImplIntfMethod.getArguments();
        if (!moduleIntfMethodArgs.get(0).getType().hasInterface(moduleImplIntfListenerClass)) {
            return false;
        }
        for (int i = 1; i < moduleIntfMethodArgs.size(); i++) {
            final SgArgument m1arg = moduleIntfMethodArgs.get(i);
            final SgArgument m2arg = moduleImplIntfMethodArgs.get(i - 1);
            if (!m1arg.getType().equals(m2arg.getType())) {
                return false;
            }
        }
        return true;
    }

    private SgClass getModuleImplIntfListener(final SgClass moduleImplIntf, final SgMethod method) {
        final List<SgMethod> methods = moduleImplIntf.getMethods();
        for (int i = 0; i < methods.size(); i++) {
            final SgMethod meth = methods.get(i);
            if (meth.getName().equals(method.getName())
                    && meth.getReturnType().equals(SgClass.VOID)
                    && isSameArgumentsPlusListener(meth, method)) {
                return meth.getArguments().get(0).getType();
            }
        }
        // Nothing found so far - Check super interfaces
        final List<SgClass> interfaces = moduleImplIntf.getInterfaces();
        for (int i = 0; i < interfaces.size(); i++) {
            final SgClass cl = getModuleImplIntfListener(interfaces.get(i), method);
            if (cl != null) {
                return cl;
            }
        }
        return null;
    }

    private void checkModuleImplIntfListenerNotNull(final SgClass listener,
            final SgClass moduleImplIntf, final SgMethod method) {
        if (listener == null) {
            final StringBuffer expected = new StringBuffer("public void ");
            expected.append(method.getName());
            expected.append("(");
            expected.append(SgUtils.firstCharUpper(method.getName()) + "Listener listener");
            final List<SgArgument> args = method.getArguments();
            for (int i = 0; i < args.size(); i++) {
                final SgArgument arg = args.get(i);
                expected.append(", ");
                expected.append(arg.getType().getName());
                expected.append(" ");
                expected.append(arg.getName());

            }
            expected.append(")");
            throw new IllegalArgumentException("No method '" + expected + "' found in interface '"
                    + moduleImplIntf.getName() + "'!");
        }
    }

    private Class<?> getListenerClass(final SgClass moduleImplIntfListener) {
        try {
            return Class.forName(moduleImplIntfListener.getName());
        } catch (final ClassNotFoundException ex) {
            throw new RuntimeException(ex);
        }
    }

    private Class getListenerImpl(final SgClass modulImplClasz, final SgClass moduleImplIntfListener) {

        final String name = modulImplClasz.getSimpleName() + "_"
                + moduleImplIntfListener.getSimpleName().replace('$', '_');
        try {
            final Class cl = Class.forName(name);
            if (LOG.isDebugEnabled()) {
                LOG.debug(name + " already exists");
            }
            return cl;
        } catch (final ClassNotFoundException ex) {
            // OK, we need to create it for the first time
            if (LOG.isDebugEnabled()) {
                LOG.debug(name + " does not exists");
            }

            final Class intfClass = getListenerClass(moduleImplIntfListener);
            final SgClass superClass = SgClass.create(pool, AbstractModuleImplIntfListener.class);
            final SgClass clasz = factory.create(modulImplClasz.getPackageName(), name, superClass,
                    null, listenerFactoryListener, intfClass);
            clasz.addInterface(moduleImplIntfListenerClass);
            if (LOG.isTraceEnabled()) {
                LOG.trace("Created ModuleImplIntfListener\n" + clasz);
            }
            final Class cl = generator.createClass(clasz);
            if (LOG.isDebugEnabled()) {
                LOG.debug(name + " was created successfully");
            }
            return cl;
        }
    }

    /**
     * {@inheritDoc}
     */
    public final List<String> createBody(final SgMethod method, final Class<?>... intf) {

        final List<String> lines = new ArrayList<String>();
        Utils.addMethodCallLogStmt(pool, lines, method);
        if (method.hasAnnotation(WaitForUserInput.class.getName())) {
            addSrcWaitingForUserInput(lines, method, moduleImplIntfClass);
        } else {
            addSrcNotWaitingForUserInput(lines, method, moduleImplIntfClass);
        }
        return lines;
    }

    private void addSrcNotWaitingForUserInput(final List<String> lines, final SgMethod method,
            final SgClass moduleImplIntf) {

        lines.add(moduleImplIntf.getName() + " module = (" + moduleImplIntf.getName()
                + ") getModule();");

        final String call;
        if (method.getArguments().size() == 0) {
            call = "module." + method.getName() + "()";
        } else {
            call = "module." + method.getName() + "(" + method.getCommaSeparatedArgumentNames()
                    + ")";
        }
        if (method.getReturnType().equals(SgClass.VOID)) {
            lines.add(call + ";");
        } else {
            addReturnValue(lines, method, call);
        }

    }

    private void addReturnValue(final List<String> lines, final SgMethod method, final String call) {
        if (method.getReturnType().isPrimitive()) {
            final SgClass nonPrimitiveClass = SgClass.getNonPrimitiveClass(pool, method
                    .getReturnType());
            final String convMethod = SgClass.getToPrimitiveMethod(nonPrimitiveClass);
            lines.add(method.getReturnType().getName() + " retVal = (("
                    + nonPrimitiveClass.getName() + ") " + call + ")." + convMethod + "();");
        } else {
            lines.add(method.getReturnType().getName() + " retVal = " + call + ";");
        }
        Utils.addVarTraceStmt(pool, lines, "retVal", method.getReturnType());
        lines.add("return retVal;");
    }

    private void addSrcWaitingForUserInput(final List<String> lines, final SgMethod method,
            final SgClass moduleImplIntf) {

        final SgClass moduleImplIntfListener = getModuleImplIntfListener(moduleImplIntf, method);
        if (LOG.isDebugEnabled()) {
            LOG.debug("moduleImplIntfListener=" + moduleImplIntfListener.getName());
        }
        checkModuleImplIntfListenerNotNull(moduleImplIntfListener, moduleImplIntf, method);

        final Class listenerImplClass = getListenerImpl(method.getOwner(), moduleImplIntfListener);

        lines.add(moduleImplIntf.getName() + " module = (" + moduleImplIntf.getName()
                + ") getModule();");
        lines.add(listenerImplClass.getName() + " listener = new " + listenerImplClass.getName()
                + "();");
        if (method.getArguments().size() == 0) {
            lines.add("module." + method.getName() + "(listener);");
        } else {
            lines.add("module." + method.getName() + "(listener, "
                    + method.getCommaSeparatedArgumentNames() + ");");
        }
        lines.add("waitForResults(listener);");
        final List<SgClass> exceptions = method.getExceptions();
        for (int i = 0; i < exceptions.size(); i++) {
            final String getter = "getFailure" + exceptions.get(i).getSimpleName() + "()";
            lines.add("if (listener." + getter + "!=null) {");
            lines.add("    throw listener." + getter + ";");
            lines.add("}");
        }
        if (!method.getReturnType().equals(SgClass.VOID)) {
            final String call = "listener.getSuccess" + method.getReturnType().getSimpleName()
                    + "()";
            addReturnValue(lines, method, call);
        }

    }

}
