/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4swing;

import java.awt.Container;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;

import javax.swing.JFrame;

import org.fuin.apps4j.Controller;
import org.fuin.apps4j.Module;
import org.fuin.apps4j.ModuleImplIntf;
import org.fuin.apps4j.View;
import org.fuin.apps4j.WindowManager;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.utils4j.Utils4J;
import org.fuin.utils4swing.common.ScreenCenterPositioner;
import org.fuin.utils4swing.common.Utils4Swing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Connects a controller that is also a module and it's view.
 * 
 * @param <MODULE>
 *            Type of {@link Module}.
 * @param <CTRL>
 *            Type of {@link Controller}.
 * @param <VIEW>
 *            Type of {@link View}.
 */
public class ModuleControllerViewConnector<MODULE extends Module, CTRL extends Controller<VIEW>, VIEW extends View<CTRL>>
        extends AbstractControllerViewConnector<CTRL, VIEW> {

    private static final Logger LOG = LoggerFactory.getLogger(ModuleControllerViewConnector.class);

    private final String modulePkg;

    private MODULE module;

    /**
     * Constructor with all possible values.
     * 
     * @param name
     *            Basic name that is shared by controller and view. Used as
     *            prefix for the generated classes - Cannot be <code>null</code>
     *            or empty.
     * @param ctrlPkg
     *            Package of the generated controller classes - Cannot be
     *            <code>null</code>.
     * @param ctrlIntf
     *            Controller interface to generate an implementation for -
     *            Cannot be <code>null</code>.
     * @param ctrlImpl
     *            The "real" controller implementation - Cannot be
     *            <code>null</code>.
     * @param viewPkg
     *            Package of the generated view classes - Cannot be
     *            <code>null</code>.
     * @param viewIntf
     *            View interface to generate an implementation for - Cannot be
     *            <code>null</code>.
     * @param viewImpl
     *            The "real" view implementation - Cannot be <code>null</code>.
     * @param modulePkg
     *            Package of the generated module classes - Cannot be
     *            <code>null</code>.
     * @param moduleIntf
     *            Module interface the controller queue implements additionally
     *            to the {@link #ctrlIntf} - Cannot be <code>null</code>.
     * @param executorService
     *            Executor service to use - Cannot be <code>null</code>.
     * @param queueCapacity
     *            Capacity of the {@link ArrayBlockingQueue}.
     */
    public ModuleControllerViewConnector(final String name, final String ctrlPkg,
            final Class<?> ctrlIntf, final CTRL ctrlImpl, final String viewPkg,
            final Class<?> viewIntf, final VIEW viewImpl, final String modulePkg,
            final Class<?> moduleIntf, final ExecutorService executorService,
            final int queueCapacity) {

        super(name, ctrlPkg, ctrlIntf, ctrlImpl, viewPkg, viewIntf, viewImpl, moduleIntf,
                executorService, queueCapacity);

        Utils4J.checkNotNull("modulePkg", modulePkg);
        this.modulePkg = modulePkg;

        Utils4J.checkNotNull("moduleIntf", moduleIntf);

    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected final SgClass createControllerQueueClass() {
        final ControllerQueueFactoryListener listener = new ControllerQueueFactoryListener(
                getPool(), getGenerator());
        final SgClass parentClass = SgClass.create(getPool(), AbstractControllerQueue.class);
        return getFactory().create(getCtrlPkg(), getName() + "ControllerQueue", parentClass, null,
                listener, getCtrlIntf(), getModuleImplIntf());
    }

    @SuppressWarnings("unchecked")
    private MODULE createModule() {
        final SgClass superClass = SgClass.create(getPool(), AbstractModuleImpl.class);
        final SgClass clasz = getFactory().create(modulePkg, getName() + "ModuleImpl", superClass,
                null, new ModuleImplFactoryListener(getPool(), getModuleImplIntf()),
                getModuleIntf());
        if (LOG.isTraceEnabled()) {
            LOG.trace("Created ModuleImpl\n" + clasz);
        }
        final MODULE module = (MODULE) getGenerator().createInstance(clasz);
        final WindowManager windowManager = createWindowManager(module);
        ((AbstractModuleImpl) module).setModule((ModuleImplIntf) getControllerQueue());
        module.setWindowManager(windowManager);
        return module;
    }

    private static WindowManager createWindowManager(final Module module) {
        return new WindowManager() {

            private JFrame frame;

            public void open() {
                if (frame != null) {
                    throw new IllegalStateException("Method 'open()' called twice!");
                }
                final Container container = (Container) module.getModuleUI();
                frame = Utils4Swing.createShowAndPosition(module.getModuleTitle(), container,
                        false, new ScreenCenterPositioner());
                // TODO Set default button
                // frame.getRootPane().setDefaultButton(container.getDefaultButton());
                frame.addWindowListener(new WindowAdapter() {
                    @Override
                    public void windowClosing(final WindowEvent e) {
                        module.cancel();
                    }
                });
            }

            public void close() {
                if (frame == null) {
                    throw new IllegalStateException("Method 'close()' called twice!");
                }
                frame.setVisible(false);
                frame = null;
            }

        };
    }

    /**
     * Returns the generated module implementation.
     * 
     * @return Module.
     */
    public final MODULE getModule() {
        if (module == null) {
            module = createModule();
        }
        return module;
    }

}
