/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4swing;

import java.lang.reflect.InvocationTargetException;

import javax.swing.SwingUtilities;

import org.fuin.utils4j.Utils4J;

/**
 * Calls a method on a target object in the EDT thread.
 * 
 * @param <RESULT>
 *            Type of result returned by the target method.
 */
public final class EDTInvoker<RESULT> {

	private final EDTGetter<RESULT> target;

	/**
	 * Constructor with target object.
	 * 
	 * @param target
	 *            Object's {@link EDTGetter#get()} method is called in the EDT
	 *            thread.
	 */
	public EDTInvoker(final EDTGetter<RESULT> target) {
		super();
		Utils4J.checkNotNull("target", target);
		this.target = target;
	}

	/**
	 * Calls the {@link EDTGetter#get()} using
	 * {@link SwingUtilities#invokeAndWait(Runnable)}.
	 * 
	 * @return Result of the method call.
	 */
	public final RESULT invoke() {
		try {
			final EDTResult<RESULT> result = new EDTResult<RESULT>();
			SwingUtilities.invokeAndWait(new Runnable() {
				public void run() {
					result.setResultValue(target.get());
				}
			});
			return result.getResultValue();
		} catch (final InterruptedException ex) {
			throw new RuntimeException(ex);
		} catch (final InvocationTargetException ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Executes a runnable using {@link SwingUtilities#invokeAndWait(Runnable)}.
	 * 
	 * @param runnable
	 *            Runnable to run.
	 */
	public static void invokeAndWait(final Runnable runnable) {
		try {
			SwingUtilities.invokeAndWait(runnable);
		} catch (final InterruptedException ex) {
			throw new RuntimeException(ex);
		} catch (final InvocationTargetException ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Executes a runnable using {@link SwingUtilities#invokeLater(Runnable)}.
	 * 
	 * @param runnable
	 *            Runnable to run.
	 */
	public static void invokeLater(final Runnable runnable) {
		SwingUtilities.invokeLater(runnable);
	}

}
