/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4swing;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;

import org.fuin.apps4j.View;

/**
 * Executes tasks submitted for a controller in sequential order. During the
 * execution the view of the controller will be blocked.
 * 
 * @param <CONTROLLER>
 *            Type of controller the task executor acts on.
 */
public final class ControllerTaskExecutor<CONTROLLER> implements Runnable {

	private volatile boolean canceled = false;

	private ArrayBlockingQueue<Runnable> events;

	private ExecutorService executor;

	private View<CONTROLLER> view;

	/**
	 * Default constructor.
	 */
	public ControllerTaskExecutor() {
		super();
	}

	/**
	 * Returns if the execution has been canceled.
	 * 
	 * @return If the task has been canceled <code>true</code> else
	 *         <code>false</code>.
	 */
	public final boolean isCanceled() {
		return canceled;
	}

	/**
	 * Cancel the task.
	 */
	public final void cancel() {
		this.canceled = true;
	}

	/**
	 * Executes the task.
	 * 
	 * @param event
	 *            Event to handle.
	 */
	public final synchronized void execute(final Runnable event) {
		if (events.isEmpty()) {
			events.add(event);
			executor.execute(this);
		} else {
			events.add(event);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public final void run() {
		view.block();
		try {
			while (!canceled && !executor.isShutdown() && !events.isEmpty()) {
				try {
					events.take().run();
				} catch (final InterruptedException ex) {
					// TODO Auto-generated catch block
					ex.printStackTrace();
				}
			}
		} finally {
			view.unblock();
		}
	}

	/**
	 * Set an event queue.
	 * 
	 * @param events
	 *            Event queue to use.
	 */
	public final void setEvents(final ArrayBlockingQueue<Runnable> events) {
		this.events = events;
	}

	/**
	 * Set an executor service.
	 * 
	 * @param executor
	 *            Executor to use.
	 */
	public final void setExecutorService(final ExecutorService executor) {
		this.executor = executor;
	}

	/**
	 * Set the view to block during task execution.
	 * 
	 * @param view
	 *            View
	 */
	public final void setView(final View<CONTROLLER> view) {
		this.view = view;
	}

}
