/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4swing;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;

import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgArgument;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.SgField;
import org.fuin.srcgen4javassist.SgMethod;
import org.fuin.srcgen4javassist.SgUtils;
import org.fuin.srcgen4javassist.SgVariable;
import org.fuin.srcgen4javassist.factory.ImplementationFactoryListener;
import org.fuin.srcgen4javassist.factory.VarListImplementationFactory;
import org.fuin.utils4j.Utils4J;
import org.slf4j.Logger;

/**
 * Creates the method bodies for a {@link AbstractControllerQueue}-Implementation.
 */
public final class ControllerQueueFactoryListener implements ImplementationFactoryListener {

    private final SgClassPool pool;

    private final ByteCodeGenerator generator;

    private final VarListImplementationFactory varListFactory;

    /**
     * Constructor with all necessary values.
     * 
     * @param pool
     *            Class pool.
     * @param generator
     *            Byte code generator.
     */
    public ControllerQueueFactoryListener(final SgClassPool pool, final ByteCodeGenerator generator) {
        super();

        Utils4J.checkNotNull("pool", pool);
        this.pool = pool;

        Utils4J.checkNotNull("generator", generator);
        this.generator = generator;

        this.varListFactory = new VarListImplementationFactory(pool);
    }

    /**
     * {@inheritDoc}
     */
    public final void afterClassCreated(final SgClass clasz) {

        final SgClass sl4jLoggerClasz = SgClass.create(pool, Logger.class);
        clasz.addField(new SgField(clasz, "private static", sl4jLoggerClasz, "LOG",
                "org.slf4j.LoggerFactory.getLogger(" + clasz.getName() + ".class)"));

    }

    /**
     * {@inheritDoc}
     */
    public final List<String> createBody(final SgMethod method, final Class<?>... intf) {

        final List<String> lines = new ArrayList<String>();

        lines.add("if (LOG.isDebugEnabled()) {");
        lines.add("    LOG.debug(\"" + method.getTypeSignature() + "\");");
        lines.add("}");
        final List<SgArgument> args = method.getArguments();
        if (args.size() > 0) {
            lines.add("if (LOG.isTraceEnabled()) {");
            for (int i = 0; i < args.size(); i++) {
                final SgArgument arg = args.get(i);
                Utils.addVarTraceStmt(pool, lines, arg.getName(), arg.getType());
            }
            lines.add("}");
        }

        final SgClass callableClass = createCallableClass(intf[0], method);
        lines.add(intf[0].getName() + " ctrl = ((" + intf[0].getName() + ") getController());");
        if (method.getArguments().size() == 0) {
            lines.add(Callable.class.getName() + " callable = new " + callableClass.getSourceName()
                    + "(ctrl);");
        } else {
            lines.add(Callable.class.getName() + " callable = new " + callableClass.getSourceName()
                    + "(ctrl, " + method.getCommaSeparatedArgumentNames() + ");");
        }
        if (method.getReturnType().equals(SgClass.VOID)) {
            lines.add("execute(callable);");
        } else {
            lines.add("Object retVal = execute(callable);");
            Utils.addVarTraceStmt(pool, lines, "retVal", method.getReturnType());
            if (method.getReturnType().isPrimitive()) {
                final SgClass nonPrimitiveClass = SgClass.getNonPrimitiveClass(pool, method
                        .getReturnType());
                final String convMethod = SgClass.getToPrimitiveMethod(nonPrimitiveClass);
                lines.add("return ((" + nonPrimitiveClass.getName() + ") retVal)." + convMethod
                        + "();");
            } else {
                lines.add("return (" + method.getReturnType().getName() + ") retVal;");
            }
        }

        return lines;

    }

    private SgClass createCallableClass(final Class<?> intf, final SgMethod method) {
        final List<SgVariable> callableArgs = new ArrayList<SgVariable>();
        callableArgs.add(new SgVariable("", SgClass.create(pool, intf), "ctrl") {
        });
        callableArgs.addAll(method.getArguments());
        final String implPackage = method.getOwner().getPackageName();
        final String implName = "Callable" + method.getUnderscoredNameAndTypes();
        final SgClass callableClass = varListFactory.create(implPackage, implName, null, method
                .getOwner(), callableArgs, new ImplementationFactoryListener() {
            public void afterClassCreated(final SgClass clasz) {
                // Do nothing
            }

            public List<String> createBody(final SgMethod runMethod, final Class<?>... intf) {
                final List<String> lines = new ArrayList<String>();
                if (method.getReturnType() == SgClass.VOID) {
                    lines.add("ctrl." + method.getCallSignature() + ";");
                    lines.add("return null;");
                } else {
                    if (method.getReturnType().isPrimitive()) {
                        final SgClass nonPrimitiveClass = SgClass.getNonPrimitiveClass(pool, method
                                .getReturnType());
                        lines.add("return " + nonPrimitiveClass.getName() + ".valueOf(ctrl."
                                + method.getCallSignature() + ");");
                    } else {
                        lines.add("return ctrl." + method.getCallSignature() + ";");
                    }
                }
                return lines;
            }
        }, Callable.class);

        generator.createClass(callableClass);
        return callableClass;
    }

}
