package org.fuin.apps4swing;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;

import org.fuin.apps4j.Controller;
import org.fuin.apps4j.ModulImplIntfRef;
import org.fuin.apps4j.ModuleImplIntf;
import org.fuin.apps4j.View;
import org.fuin.srcgen4javassist.ByteCodeGenerator;
import org.fuin.srcgen4javassist.SgClass;
import org.fuin.srcgen4javassist.SgClassPool;
import org.fuin.srcgen4javassist.factory.ImplementationFactory;
import org.fuin.utils4j.Utils4J;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractControllerViewConnector<CTRL extends Controller<VIEW>, VIEW extends View<CTRL>> {

    private static final Logger LOG = LoggerFactory
            .getLogger(AbstractControllerViewConnector.class);

    private final ByteCodeGenerator generator;
    private final SgClassPool pool;
    private final ImplementationFactory factory;
    private final ExecutorService executorService;
    private final String name;
    private final String ctrlPkg;
    private final Class<?> ctrlIntf;
    private final CTRL ctrlImpl;
    private final String viewPkg;
    private final Class<?> viewIntf;
    private final VIEW viewImpl;
    private final Class<?> moduleIntf;
    private final Class<? extends ModuleImplIntf> moduleImplIntf;
    private final int queueCapacity;

    private VIEW toViewDispatcher;
    private CTRL controllerQueue;
    private CTRL toControllerDispatcher;

    /**
     * Constructor with all possible values.
     * 
     * @param name
     *            Basic name that is shared by controller and view. Used as
     *            prefix for the generated classes - Cannot be <code>null</code>
     *            or empty.
     * @param ctrlPkg
     *            Package of the generated controller classes - Cannot be
     *            <code>null</code>.
     * @param ctrlIntf
     *            Controller interface to generate an implementation for -
     *            Cannot be <code>null</code>.
     * @param ctrlImpl
     *            The "real" controller implementation - Cannot be
     *            <code>null</code>.
     * @param viewPkg
     *            Package of the generated view classes - Cannot be
     *            <code>null</code>.
     * @param viewIntf
     *            View interface to generate an implementation for - Cannot be
     *            <code>null</code>.
     * @param viewImpl
     *            The "real" view implementation - Cannot be <code>null</code>.
     * @param moduleIntf
     *            Module interface the controller queue implements additionally
     *            to the {@link #ctrlIntf} or <code>null</code>.
     * @param executorService
     *            Executor service to use - Cannot be <code>null</code>.
     * @param queueCapacity
     *            Capacity of the {@link ArrayBlockingQueue}.
     */
    public AbstractControllerViewConnector(final String name, final String ctrlPkg,
            final Class<?> ctrlIntf, final CTRL ctrlImpl, final String viewPkg,
            final Class<?> viewIntf, final VIEW viewImpl, final Class<?> moduleIntf,
            final ExecutorService executorService, final int queueCapacity) {

        super();

        Utils4J.checkNotNull("name", name);
        Utils4J.checkNotEmpty("name", name);
        this.name = name;

        Utils4J.checkNotNull("ctrlPkg", ctrlPkg);
        this.ctrlPkg = ctrlPkg;
        Utils4J.checkNotNull("ctrlIntf", ctrlIntf);
        this.ctrlIntf = ctrlIntf;
        Utils4J.checkNotNull("ctrlImpl", ctrlImpl);
        this.ctrlImpl = ctrlImpl;

        Utils4J.checkNotNull("viewPkg", viewPkg);
        this.viewPkg = viewPkg;
        Utils4J.checkNotNull("viewIntf", viewIntf);
        this.viewIntf = viewIntf;
        Utils4J.checkNotNull("viewImpl", viewImpl);
        this.viewImpl = viewImpl;

        this.moduleIntf = moduleIntf;

        if (moduleIntf == null) {
            moduleImplIntf = null;
        } else {
            final ModulImplIntfRef ref = moduleIntf.getAnnotation(ModulImplIntfRef.class);
            if (ref == null) {
                throw new IllegalArgumentException("The interface '" + moduleIntf.getName()
                        + "' ist not annotated with '" + ModulImplIntfRef.class.getName() + "'!");
            }
            moduleImplIntf = ref.value();
        }

        Utils4J.checkNotNull("executorService", executorService);
        this.executorService = executorService;
        this.queueCapacity = queueCapacity;

        generator = new ByteCodeGenerator();
        pool = new SgClassPool();
        factory = new ImplementationFactory(pool);

        toViewDispatcher = createToViewDispatcher();
        controllerQueue = createControllerQueue(toViewDispatcher);
        toControllerDispatcher = createToControllerDispatcher(controllerQueue);

        ctrlImpl.setView(toViewDispatcher);
        viewImpl.setController(toControllerDispatcher);

    }

    @SuppressWarnings("unchecked")
    private VIEW createToViewDispatcher() {

        final SgClass toViewDispClass = SgClass.create(pool, AbstractToViewDispatcher.class);

        final ToViewDispatcherFactoryListener listener = new ToViewDispatcherFactoryListener(pool,
                generator);

        final SgClass toViewDispClasz = factory.create(viewPkg, name + "ToViewDispatcher",
                toViewDispClass, null, listener, viewIntf);

        if (LOG.isTraceEnabled()) {
            LOG.trace("Created ToViewDispatcher\n" + toViewDispClasz);
        }

        final VIEW toViewDisp;
        try {
            toViewDisp = (VIEW) generator.createInstance(toViewDispClasz);
        } catch (final Throwable throwable) {
            final String msg = "Error creating ToViewDispatcher";
            LOG.error(msg + ":\n" + toViewDispClasz);
            throw new RuntimeException(msg, throwable);
        }

        final AbstractToViewDispatcher impl = (AbstractToViewDispatcher) toViewDisp;
        impl.setView(viewImpl);
        return toViewDisp;

    }

    @SuppressWarnings("unchecked")
    private CTRL createToControllerDispatcher(final CTRL ctrlQueue) {

        final SgClass toCtrlDispClass = SgClass.create(pool, AbstractToControllerDispatcher.class);

        final ToControllerDispatcherFactoryListener listener = new ToControllerDispatcherFactoryListener(
                pool, generator);

        final SgClass toCtrlDispClasz = factory.create(ctrlPkg, name + "ToControllerDispatcher",
                toCtrlDispClass, null, listener, ctrlIntf);

        if (LOG.isTraceEnabled()) {
            LOG.trace("Created ToControllerDispatcher\n" + toCtrlDispClasz);
        }

        final CTRL toCtrlDisp;
        try {
            toCtrlDisp = (CTRL) generator.createInstance(toCtrlDispClasz);
        } catch (final Throwable throwable) {
            final String msg = "Error creating ToControllerDispatcher";
            LOG.error(msg + ":\n" + toCtrlDispClasz);
            throw new RuntimeException(msg, throwable);
        }

        final AbstractToControllerDispatcher impl = (AbstractToControllerDispatcher) toCtrlDisp;
        impl.setController(ctrlQueue);
        impl.setExecutorService(executorService);

        return toCtrlDisp;

    }

    @SuppressWarnings("unchecked")
    private CTRL createControllerQueue(final VIEW toViewDispatcher) {

        final SgClass ctrlQueueClasz = createControllerQueueClass();

        if (LOG.isTraceEnabled()) {
            LOG.trace("Created ControllerQueue\n" + ctrlQueueClasz);
        }

        final CTRL ctrlTaskDispImpl;
        try {
            ctrlTaskDispImpl = (CTRL) generator.createInstance(ctrlQueueClasz);
        } catch (final Throwable throwable) {
            final String msg = "Error creating ControllerQueue";
            LOG.error(msg + ":\n" + ctrlQueueClasz);
            throw new RuntimeException(msg, throwable);
        }

        final AbstractControllerQueue impl = (AbstractControllerQueue) ctrlTaskDispImpl;
        impl.setController(ctrlImpl);

        final ControllerTaskExecutor ctrlTaskExecutor = new ControllerTaskExecutor();
        ctrlTaskExecutor.setEvents(new ArrayBlockingQueue(queueCapacity));
        ctrlTaskExecutor.setExecutorService(executorService);
        ctrlTaskExecutor.setView(toViewDispatcher);
        impl.setControllerTaskExecutor(ctrlTaskExecutor);

        return ctrlTaskDispImpl;

    }

    /**
     * Returns the executor service used for the generated implementations.
     * 
     * @return Executor.
     */
    public final ExecutorService getExecutorService() {
        return executorService;
    }

    /**
     * Returns the class pool.
     * 
     * @return Class pool used.
     */
    protected final SgClassPool getPool() {
        return pool;
    }

    /**
     * Returns the byte code generator used internally.
     * 
     * @return Byte code generator.
     */
    protected final ByteCodeGenerator getGenerator() {
        return generator;
    }

    /**
     * Returns the implementation factory used internally.
     * 
     * @return Implementation factory.
     */
    protected final ImplementationFactory getFactory() {
        return factory;
    }

    /**
     * Returns the basic name that is shared by controller and view. Used as
     * prefix for the generated classes.
     * 
     * @return Name - Never <code>null</code> or empty
     */
    public final String getName() {
        return name;
    }

    /**
     * Returns the package of the generated controller classes.
     * 
     * @return Package - Never <code>null</code>.
     */
    public final String getCtrlPkg() {
        return ctrlPkg;
    }

    /**
     * Returns the controller interface to generate an implementation for.
     * 
     * @return Controller interface - Never <code>null</code>.
     */
    protected final Class<?> getCtrlIntf() {
        return ctrlIntf;
    }

    /**
     * Returns the "real" controller implementation.
     * 
     * @return Controller implementation - Never <code>null</code>.
     */
    protected final CTRL getCtrlImpl() {
        return ctrlImpl;
    }

    /**
     * Returns the package of the generated view classes.
     * 
     * @return Package - Never <code>null</code>.
     */
    protected final String getViewPkg() {
        return viewPkg;
    }

    /**
     * Returns the view interface to generate an implementation for.
     * 
     * @return View interface - Never <code>null</code>.
     */
    protected final Class<?> getViewIntf() {
        return viewIntf;
    }

    /**
     * Returns the "real" view implementation.
     * 
     * @return View implementation - Never <code>null</code>.
     */
    protected final VIEW getViewImpl() {
        return viewImpl;
    }

    /**
     * Returns the module interface.
     * 
     * @return Interface or <code>null</code>.
     */
    protected final Class<?> getModuleIntf() {
        return moduleIntf;
    }

    /**
     * Returns the module implementation interface.
     * 
     * @return Interface or <code>null</code>.
     */
    protected final Class<? extends ModuleImplIntf> getModuleImplIntf() {
        return moduleImplIntf;
    }

    /**
     * Returns the capacity of the {@link ArrayBlockingQueue}.
     * 
     * @return Capacity.
     */
    protected final int getQueueCapacity() {
        return queueCapacity;
    }

    /**
     * Returns the generated "to view dispatcher" implementation.
     * 
     * @return Dispatcher.
     */
    public final VIEW getToViewDispatcher() {
        return toViewDispatcher;
    }

    /**
     * Returns the generated "to controller dispatcher" implementation.
     * 
     * @return Dispatcher.
     */
    public final CTRL getToControllerDispatcher() {
        return toControllerDispatcher;
    }

    /**
     * Returns the generated controller queue implementation.
     * 
     * @return Controller queue.
     */
    public final CTRL getControllerQueue() {
        return controllerQueue;
    }

    /**
     * Create the appropriate queue class.
     * 
     * @return Class.
     */
    protected abstract SgClass createControllerQueueClass();

}
