/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4swing;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;

/**
 * Queues commands for the controller and executes them in sequential order.
 * 
 * @param <CONTROLLER>
 *            Type of the controller.
 * @param <VIEW>
 *            Type of the view.
 */
public abstract class AbstractControllerQueue<CONTROLLER, VIEW> {

	private ControllerTaskExecutor<CONTROLLER> ctrlTaskExecutor;

	private CONTROLLER ctrl;

	/**
	 * Default constructor.
	 */
	public AbstractControllerQueue() {
		super();
	}

	/**
	 * Executes a task and returns the result of the command.
	 * 
	 * @param callable
	 *            Task to execute.
	 * 
	 * @return Result of the operation.
	 */
	protected final Object execute(final Callable<Object> callable) {
		try {
			final FutureTask<Object> ft = new FutureTask<Object>(callable);
			ctrlTaskExecutor.execute(ft);
			return ft.get();
		} catch (final InterruptedException ex) {
			// TODO Handling!?
			throw new RuntimeException(ex);
		} catch (final ExecutionException ex) {
			// TODO Handling!?
			throw new RuntimeException(ex.getCause());
		}
	}

	/**
	 * Sets the executor that actually does the work.
	 * 
	 * @param ctrlTaskExecutor
	 *            Task executor to set.
	 */
	public final void setControllerTaskExecutor(
			final ControllerTaskExecutor<CONTROLLER> ctrlTaskExecutor) {
		this.ctrlTaskExecutor = ctrlTaskExecutor;
	}

	/**
	 * Returns the controller that contains the real application logic.
	 * 
	 * @return Controller or <code>null</code>.
	 */
	public final CONTROLLER getController() {
		return this.ctrl;
	}

	/**
	 * Sets the controller that contains the real application logic.
	 * 
	 * @param ctrl
	 *            Controller to set or <code>null</code>.
	 */
	public final void setController(final CONTROLLER ctrl) {
		this.ctrl = ctrl;
	}

}
