/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4j.menu;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementRef;
import javax.xml.bind.annotation.XmlElementRefs;
import javax.xml.bind.annotation.XmlRootElement;

import org.fuin.apps4j.base.Target;
import org.fuin.apps4j.base.TargetImpl;

/**
 * Default implementation of the menu item.
 */
@XmlRootElement(name = "item")
@XmlAccessorType(XmlAccessType.FIELD)
public final class TextMenuItemImpl implements TextMenuItem {

    @XmlAttribute
    private final String text;

    @XmlAttribute(name = "mnemonic-index")
    private final Integer mnemonicIndex;

    @XmlAttribute
    private final String accelerator;

    @XmlElement(type = TargetImpl.class)
    private final Target target;

    @XmlElementRefs( { @XmlElementRef(type = TextMenuItemImpl.class),
            @XmlElementRef(type = DividerMenuItem.class) })
    private final List<MenuItem> items;

    /**
     * Default constructor for JAXB.
     */
    TextMenuItemImpl() {
        super();
        text = null;
        mnemonicIndex = null;
        accelerator = null;
        target = null;
        items = null;
    }

    /**
     * Constructor with text.
     * 
     * @param text
     *            Text.
     */
    public TextMenuItemImpl(final String text) {
        this(text, null, null, null);
    }

    /**
     * Constructor with text and mnemonic index.
     * 
     * @param text
     *            Text.
     * @param mnemonicIndex
     *            Index.
     */
    public TextMenuItemImpl(final String text, final Integer mnemonicIndex) {
        this(text, null, mnemonicIndex, null);
    }

    /**
     * Constructor with text, accelerator and mnemonic index.
     * 
     * @param text
     *            Text.
     * @param accelerator
     *            Keyboard accelerator.
     * @param mnemonicIndex
     *            Index.
     */
    public TextMenuItemImpl(final String text, final String accelerator, final Integer mnemonicIndex) {
        this(text, accelerator, mnemonicIndex, null);
    }

    /**
     * Constructor with text and accelerator.
     * 
     * @param text
     *            Text.
     * @param accelerator
     *            Keyboard accelerator.
     */
    public TextMenuItemImpl(final String text, final String accelerator) {
        this(text, accelerator, null, null);
    }

    /**
     * Constructor with all data.
     * 
     * @param text
     *            Text.
     * @param accelerator
     *            Keyboard accelerator.
     * @param mnemonicIndex
     *            Index.
     * @param target
     *            Target.
     * @param items
     *            Array of sub menu items. A copy of this list will be made.
     */
    public TextMenuItemImpl(final String text, final String accelerator,
            final Integer mnemonicIndex, final Target target, final MenuItem... items) {
        this(text, accelerator, mnemonicIndex, target,
                ((items == null) || (items.length == 0) ? null : Arrays.asList(items)));
    }

    /**
     * Constructor with all data.
     * 
     * @param text
     *            Text.
     * @param accelerator
     *            Keyboard accelerator.
     * @param mnemonicIndex
     *            Index.
     * @param target
     *            Target.
     * @param items
     *            List of sub menu items. A copy of this list will be made.
     */
    public TextMenuItemImpl(final String text, final String accelerator,
            final Integer mnemonicIndex, final Target target, final List<MenuItem> items) {
        super();
        this.text = text;
        this.accelerator = accelerator;
        this.mnemonicIndex = mnemonicIndex;
        this.target = target;
        if (items == null) {
            this.items = null;
        } else {
            this.items = new ArrayList<MenuItem>(items.size());
            this.items.addAll(items);
        }
    }

    @Override
    public final String getAccelerator() {
        return accelerator;
    }

    @Override
    public final Integer getMnemonicIndex() {
        return mnemonicIndex;
    }

    @Override
    public final String getText() {
        return text;
    }

    @Override
    public final List<MenuItem> getItems() {
        if (items == null) {
            return null;
        }
        return Collections.unmodifiableList(items);
    }

    @Override
    public final Target getTarget() {
        return target;
    }

    @Override
    public final String toString() {
        return text;
    }

}
