/**
 * Copyright (C) 2009 Future Invent Informationsmanagement GmbH. All rights
 * reserved. <http://www.fuin.org/>
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */
package org.fuin.apps4j.menu;

import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElementRef;
import javax.xml.bind.annotation.XmlElementRefs;
import javax.xml.bind.annotation.XmlRootElement;

import org.fuin.apps4j.base.TargetImpl;

/**
 * Default implementation of the menu.
 */
@XmlRootElement(name = "menu")
@XmlAccessorType(XmlAccessType.FIELD)
public final class MenuImpl implements Menu {

    @XmlAttribute(required = true)
    private final String id;

    @XmlElementRefs( { @XmlElementRef(type = TextMenuItemImpl.class),
            @XmlElementRef(type = DividerMenuItem.class) })
    private final List<MenuItem> items;

    /**
     * Default constructor for JAXB.
     */
    MenuImpl() {
        super();
        this.id = null;
        this.items = null;
    }

    /**
     * Constructor with all data.
     * 
     * @param id
     *            Unique id.
     * @param items
     *            Array of items - Copied to avoid external changes.
     */
    public MenuImpl(final String id, final MenuItem... items) {
        this(id, ((items == null) || (items.length == 0) ? null : Arrays.asList(items)));
    }

    /**
     * Constructor with all data.
     * 
     * @param id
     *            Unique id.
     * @param items
     *            List of items - Copied to avoid external changes.
     */
    public MenuImpl(final String id, final List<MenuItem> items) {
        super();
        if (id == null) {
            throw new IllegalArgumentException("The argument 'id' cannot be null!");
        }
        this.id = id;
        if (items == null) {
            this.items = null;
        } else {
            this.items = new ArrayList<MenuItem>(items.size());
            this.items.addAll(items);
        }
    }

    @Override
    public final String getId() {
        return id;
    }

    @Override
    public final List<MenuItem> getItems() {
        if (items == null) {
            return null;
        }
        return Collections.unmodifiableList(items);
    }

    /**
     * Creates a menu from XML.
     * 
     * @param xml
     *            XML to create a menu from.
     * 
     * @return Menu.
     * 
     * @throws UnmarshalMenuException
     *             Error reading the menu.
     */
    public static MenuImpl createFromXml(final String xml) throws UnmarshalMenuException {
        return createFromXml(new StringReader(xml));
    }

    /**
     * Creates a menu from XML.
     * 
     * @param reader
     *            Reader with XML to create a menu from.
     * 
     * @return Menu.
     * 
     * @throws UnmarshalMenuException
     *             Error reading the menu.
     */
    public static MenuImpl createFromXml(final Reader reader) throws UnmarshalMenuException {
        try {
            final JAXBContext jaxbContext = createJaxbContext();
            final Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
            return (MenuImpl) unmarshaller.unmarshal(reader);
        } catch (final JAXBException ex) {
            throw new UnmarshalMenuException(ex);
        }
    }

    /**
     * Marshals the menu to an XML String.
     * 
     * @return XML String.
     * 
     * @throws MarshalMenuException
     *             Error writing the menu.
     */
    public String writeToXml() throws MarshalMenuException {
        return writeToXml(this);
    }

    /**
     * Marshals the menu to XML.
     * 
     * @param writer
     *            Writer to write the menu to.
     * 
     * @throws MarshalMenuException
     *             Error writing the menu.
     */
    public void writeToXml(final Writer writer) throws MarshalMenuException {
        writeToXml(this, writer);
    }

    /**
     * Marshals the menu to an XML String.
     * 
     * @param menu
     *            Menu to write as XML.
     * 
     * @return XML String.
     * 
     * @throws MarshalMenuException
     *             Error writing the menu.
     */
    private static String writeToXml(final MenuImpl menu) throws MarshalMenuException {
        final StringWriter writer = new StringWriter();
        writeToXml(menu, writer);
        return writer.toString();
    }

    /**
     * Marshals the menu to XML.
     * 
     * @param menu
     *            Menu to write as XML.
     * @param writer
     *            Writer to write the menu to.
     * 
     * @throws MarshalMenuException
     *             Error writing the menu.
     */
    private static void writeToXml(final MenuImpl menu, final Writer writer)
            throws MarshalMenuException {
        try {
            final JAXBContext jaxbContext = createJaxbContext();
            final Marshaller marshaller = jaxbContext.createMarshaller();
            marshaller.marshal(menu, writer);
        } catch (final JAXBException ex) {
            throw new MarshalMenuException(ex);
        }
    }

    private static JAXBContext createJaxbContext() throws JAXBException {
        final JAXBContext jaxbContext = JAXBContext.newInstance(MenuImpl.class,
                TextMenuItemImpl.class, TargetImpl.class, DividerMenuItem.class);
        return jaxbContext;
    }

}
